package datalog_ra.parser;

import datalog_ra.base.relation.*;

import java.io.*;
import java.util.HashMap;

/**
 * Datalog parser. Currently only supports parsing facts, i.e. rules
 * in form 'r(a, b, c, ..).'.
 */
public class Parser {
    private Lexer lexer;
    private HashMap<String, Relation> relations;

    public Parser(InputStream input) {
        lexer = new Lexer(input);
        relations = new HashMap<>();
    }

    public HashMap<String, Relation> getRelations() {
        return relations;
    }

    private void insert(String rel, Tuple t) {
        if (!relations.containsKey(rel)) {
            relations.put(rel, new Relation());
        }
        relations.get(rel).add(t);
    }

    /**
     * Parses input stream.
     * @throws ParseException
     */
    public void parse() throws ParseException {
        // program := { rule };
        lexer.next();
        while (!is(Token.eof)) {
            parseRule();
        }
    }

    private void parseRule() throws ParseException {
        // rule = identifier, lparen, identifier, { comma, identifier }, rparen, dot;
        Tuple tuple = new Tuple();
        String relationName = readIdentifier();
        read(Token.lparen);
        while (is(Token.identifier)) {
            tuple.add(new Attribute(readIdentifier()));
            if (!is(Token.rparen)) {
                read(Token.comma);
            }
        }
        read(Token.rparen);
        read(Token.dot);
        insert(relationName, tuple);
    }

    private boolean is(Token t) {
        return lexer.getToken() == t;
    }

    private String readIdentifier() throws ParseException {
        String identStr = lexer.getIdentifierStr();
        read(Token.identifier);
        return identStr;
    }

    private void read(Token t) throws ParseException {
        if (lexer.getToken() != t) {
            throw new ParseException(lexer.getLine() + ":" + lexer.getColumn() + ": " +
                    "unexpected " + lexer.getToken().toString() + ", expected " + t.toString());
        }
        lexer.next();
    }

    public class ParseException extends Exception {
        public ParseException(String message) {
            super(message);
        }
    }
}
