import java.awt.*;
import java.awt.event.*;
import java.awt.geom.*;
import javax.swing.*;

/**
 * Trieda Plotter umoznuje vykreslovanie jednoduchych grafov. Vlastny graf sa
 * definuje v metode triedy <B>plot</B> pomocou prikazov riadenia fiktivneho
 * pera, podobne ako pri realnom suradnicovom zapisovaci.
 * Pred zacatim kreslenia je potrebne definovat zadat mierku osi grafu. Do grafu
 * je mozne vykreslit osi a mriezku grafu. 
 * Pri riadeni vykreslovania grafu je mozne presuvat pero zodvihnute
 * bez kreslenia ciary, polozene s vykreslovanim a volit farbu ciary. V jednom
 * obrazku je mozne vykreslit niekolko grafov s roznou mierkou.
 */
public class GrafickyViewer extends Canvas {
        
    // Fyzicky rozmer okna v pixloch
    int sizeY = 500;
    int sizeX = 500;
    
    // Aktualna poloha kurzoru v logickych suradniciach
    double cursorX = 0.0;
    double cursorY = 0.0;
    
    // Rozmery grafu v logickych suradniciach
    double rangeX = 0.0;
    double rangeY = 0.0;
    
    // Parametre pre definovanie logickych hranic vykreslovania
    double minX = 0.0;
    double minY = 0.0;
    double maxX = 0.0;
    double maxY = 0.0;
    
    // Fyzicke suradnice polohy logickej suradnice 0,0 
    double originX = 0.0;
    double originY = 0.0;
    
    Graphics2D g2;
    
    Prostredie miska; // parameter konstruktora = ktore prostredie sa ma vykreslovat
    boolean isNotPlotted=false;
 	
    /**
     * Konstruktor triedy. 
     * @param name meno okna 
     */
    GrafickyViewer(String name, Prostredie miska){   // do konstruktora pridany parameter = ktore prostredie sa ma vykreslovat 	
//        super(name);                    // parameter odovzdavany konstruktoru supertriedy
	this.miska=miska;    // parameter - ktore prostredie sa bude vykreslovat
	sizeX=miska.xSize+100;    
 	sizeY=miska.ySize+100;    
        setSize(sizeX,sizeY);           // definovanie velkosti kresliaceho okna
        setScale(-50,miska.xSize+50,-50,miska.ySize+50);            // skalovanie grafu je trivialne =  rozmer okna, akurat sa spravne urobi kladny smer a pridaju okraje
        setBackground(Color.white);     // nastavenie farby pozadia
        
                                        // nastavenie spracovania standardnych udalosti generovanych
                                        // operacnym systemom 
//        addWindowListener(new WindowAdapter() {
                                        // spracovanie udalosti zatvorenia okna 
                                        //   => standardne ukoncenie aplikacie
//         public void windowClosing(WindowEvent e) {System.exit(0);}
//            }); 
//       show();
    }

    /**
     * Fuknkcia paint je vyvolavana systemom vzdy pri poziadavke na obnovu grafickeho obsahu okna 
     * napr. vytvorenie okna, odokrytie a pod. Pri volani funkcie je zaroven urceny graficky kontext
     * t.j. miesto, kam je presmerovany graficky vystup.
     * @param g graficky kontext pouzity pre kreslenie
     */
    public void paint(Graphics g) {
        g2 = (Graphics2D) g;
        g2.setRenderingHint(RenderingHints.KEY_ANTIALIASING, RenderingHints.VALUE_ANTIALIAS_OFF); //na bodky netreba anti-aliasing
        plot();
    }

    public void update(Graphics g) // vyradenie neustaleho prekreslovania pozadia
    {
	paint(g);
    }    

    /**
      * Definovanie logickeho rozmeru plotovacej obrazovky (skalovanie grafu).
      * @param minX minimum hodnoty osi X
      * @param maxX maximum hodnoty osi X
      * @param minY minimum hodnoty osi Y
      * @param maxY maximum hodnoty osi Y
      */
    public void setScale(double minX, double maxX, double minY, double maxY){
        this.minX = minX;
        this.maxX = maxX;
        this.minY = maxY;       // !!! pozor - finta, toto nie je chyba
        this.maxY = minY;       // graficke suradnice su otocene 'hore nohami'
        
        this.rangeX = this.maxX - this.minX;
        this.rangeY = this.maxY - this.minY;
            
        originX = (-this.minX)*sizeX/rangeX;
        originY = (-this.minY)*sizeY/rangeY;  
    }
    
    /**
     * Vykreslenie osi na grafe 
     */
    public void drawAxis(){
        g2.draw(new Line2D.Double(originX, 0.0, originX, sizeY));
        g2.draw(new Line2D.Double(0.0, originY, sizeY, originY));
    }

    
    /**
     * Vykreslenie mriezky na grafe. velkost mriezky je konstantna
     * 10 x 10 pixlov. 
     */
    public void drawGrid(){
        g2.setColor(Color.cyan);
        for(double x=0.0; x<=sizeX; x+=10.0){
            g2.draw(new Line2D.Double(x, 0.0, x, sizeY));
        }
                
        for(double y=0.0; y<=sizeY; y+=10.0){
            g2.draw(new Line2D.Double(0.0, y, sizeX, y));
        }
    }
    
    /**
      * Presun pera na novu poziciu bez kreslenia.
      * @param x logicka hodnota X
      * @param y logicka hodnota Y
      */
    public void moveTo(double x, double y){
        cursorX = (x-minX)*sizeX/rangeX;
        cursorY = (y-minY)*sizeY/rangeY;
    }
    
    /**
      * Presun pera na novu poziciu s kreslenim.
      * @param x logicka hodnota X
      * @param y logicka hodnota Y
      */
    public void lineTo(double x, double y){
     
        double cursorNewX = (x-minX)*sizeX/rangeX;
        double cursorNewY = (y-minY)*sizeY/rangeY;
        
        g2.draw(new Line2D.Double(cursorX, cursorY, cursorNewX, cursorNewY));
        
        cursorX = cursorNewX;
        cursorY = cursorNewY;
    }
	
    /**
      * Vykreslenie znacky (stvorceka) na aktualnej pozicii pera.
      */
    public void drawPoint(){  // pre nase ucely vykreslime obdlznik s rozmerom 1 pixel 
	g2.drawRect((int)cursorX, (int)cursorY,1, 1);
     }
	
    /**
      * Volba farby pera. 
      * @param color Farba pera. K dispozicii su nasledujuce farby: Color.black, Color.blue,
      * Color.cyan, Color.darkGray, Color.gray, Color.lightGray, Color.green, Color.magenta, 
      * Color.orange, Color.pink, Color.red, Color.white, Color.yellow
      */
    public void setColor(Color color){
        g2.setColor(color);
    }
    
    /**
      * Vykreslenie textu, poloha textu je udana aktualnou poziciou pera. 
      * @param text text, ktory sa vykreli na aktualnu polohu pera
      */
    public void drawText(String text){
        g2.drawString(text,(float)cursorX,(float)cursorY);      // konverzia double->float
    }                                                           // z neznamich dovodov nie je pouzity double

    /**
     * Vo funkcii plot mozete definovat kreslenie vlastnych grafov.
     */
    public void plot(){
        isNotPlotted=true;
        for(int x=0;x<miska.xSize;x++)   // pre vsetky body misky
		for(int y=0;y<miska.ySize;y++){
			moveTo(x,y);
			int r=255*miska.getValue(x,y)/miska.levels; if(r>255) r=255;  // ak sa zmensi levels pocas vypoctu
			int g=0; if(g>255) r=255;  // ak sa zmensi levels pocas vypoctu
			int b=255; if(r>255) r=255;  // ak sa zmensi levels pocas vypoctu
			setColor(new Color(r,g,b));  // farba sa meni od modrej po magentu
			drawPoint();
		}	
	isNotPlotted=false;
    }    
}
