import tables
from flask import Flask
from flask_sqlalchemy import SQLAlchemy
from tables import Users, UserInfo, ContactTypes, UserContactVarchar, UserContactText, UserContactPhoto, \
    WorkplaceUsers, UserWorkplacePermissions, WorkplaceLangs, Workplaces, JobTitles, JobTitleLangs, \
    Publications, LastUpdate
from datetime import date, timedelta, datetime
from dotenv import dotenv_values
from _decimal import Decimal
import mysql.connector
import re
import requests
import csv

def createLocalDB(db: SQLAlchemy, app: Flask):
    """Creates empty local database.

    :return: SQLAlchemy object"""

    print("Creating local database.")
    print("Creating tables.")

    with app.app_context():
        db.create_all()

    print("Local database created")


def fetchData(tables: list[str], updated: datetime) -> dict[
        str, list[dict[str, float | int | Decimal | str | bytes | date | timedelta | datetime | set[str] | None, ...]]]:
    """Fetches data from central database.

    :param tables:
        tables from central database to be fetched

    :return: a dictionary, where tables are keys and lists of table rows (dictionaries) are values"""

    try:
        print("Creating connection.")
        config = dotenv_values(".env")
        data = {}

        connectionCDB = mysql.connector.connect(
            host=config['HOST_CDB'],
            user=config['USER_CDB'],
            password=config['PASSWORD_CDB'],
            database=config['DATABASE_CDB'])

        print("Connection created successfully.")

        cursor = connectionCDB.cursor(dictionary=True)
        for table in tables:
            print("Fetching data from table '" + table + "'.")
            cursor.execute("SELECT * FROM " + table + " WHERE " + table + ".updated>" + str(updated))
            rows = cursor.fetchall()
            data[table] = rows

        print("Closing connection.")
        connectionCDB.close()

        print("Data fetched successfully.")
        return data

    except mysql.connector.Error as err:
        print("Something went wrong: {}".format(err))


def getUserPublications(uoc: int):

    try:
        print("Creating connection.")
        config = dotenv_values(".env")

        connectionECP = mysql.connector.connect(
            host=config['HOST_EPC'],
            user=config['USER_EPC'],
            port=config['PORT_EPC'],
            password=config['PASSWORD_EPC'],
            database=config['DATABASE_EPC'])

        print("Connection created successfully.")

        cursor = connectionECP.cursor(dictionary=True)
        year = str(date.today().year)
        print("Fetching `publications` from database `epc_mirror`.")
        sql = "SELECT distinct display_autori AS autori,nazov, display AS kde, pocet_autorov, o.auth_id, " \
              "(ohlasov>=10 AND rok>" + year + "-5) AS vela_ohlasov, kodepc IN ('AAA','AAB','ABA','ABB','ACA','ACB'," \
                                               "'ADC','ADD','ADM','ADN','AFC','AFD','V1','V2','V3') as slusny_zdroj, " \
                                               "(rok>" + year + "-2) AS aktualna " \
              "FROM osoba AS o, publikacia_autor AS pa, publikacia AS p " \
              "WHERE o.uoc=" + str(uoc) + " AND o.auth_id=pa.auth_id AND pa.bib_id=p.bib_id " \
              "ORDER BY ((aktualna AND slusny_zdroj) OR vela_ohlasov) DESC,slusny_zdroj DESC,vytvorene_epc DESC " \
              "LIMIT 5;"
        cursor.execute(sql)
        data = cursor.fetchall()

        print("Closing connection.")
        connectionECP.close()

        print("Data fetched successfully.")
        return data

    except mysql.connector.Error as err:
        print("Something went wrong: {}".format(err))


def populateData(db: SQLAlchemy, app: Flask, tables: list[str]):
    """Populates data `data` into database `db`.

    :param db: database as SQLAlchemy object,
    :param app: TODO
    :param tables: list of tables in local database"""

    lastUpdateResult = db.session.query(LastUpdate).all()
    if len(lastUpdateResult) != 0:
        lastUpdate = lastUpdateResult[0].time
    else:
        lastUpdate = 0

    print("Fetching data from databases.")
    data = fetchData(tables, lastUpdate)

    print("Populating local database.")
    with app.app_context():
        for table in tables:
            if table == "users":
                print("Populating table `users`.")
                for row in data[table]:
                    datum = Users()
                    datum.id = row["id"]
                    datum.updated = row["updated"]
                    datum.login = row["login"]
                    datum.uoc = row["uoc"]
                    datum.deleted = row["deleted"]

                    # db.session.merge(datum)
                    db.session.merge(datum)
                db.session.commit()
            elif table == "user_info":
                print("Populating table `user_info`.")
                for row in data[table]:
                    datum = UserInfo()
                    datum.id = row["id"]
                    datum.user_id = row["user_id"]
                    datum.updated = row["updated"]
                    datum.first_name = row["first_name"]
                    datum.middle_name = row["middle_name"]
                    datum.last_name = row["last_name"]
                    datum.title_before = row["title_before"]
                    datum.title_after = row["title_after"]
                    datum.maiden_name = row["maiden_name"]

                    db.session.merge(datum)
                db.session.commit()
            elif table == "contact_types":
                print("Populating table `contact_types`.")
                for row in data[table]:
                    datum = ContactTypes()
                    datum.id = row["id"]
                    datum.updated = row["updated"]
                    datum.name = row["name"]
                    datum.displayable = row["displayable"]
                    datum.editable = row["editable"]
                    datum.data_type = row["data_type"]
                    datum.input_type = row["input_type"]

                    db.session.merge(datum)
                db.session.commit()
            elif table == "user_contact_varchar":
                print("Populating table `user_contact_varchar`.")
                for row in data[table]:
                    datum = UserContactVarchar()
                    datum.id = row["id"]
                    datum.updated = row["updated"]
                    datum.value = row["value"]
                    datum.visible = row["visible"]
                    datum.workplace_user_id = row["workplace_user_id"]
                    datum.contact_type_id = row["contact_type_id"]
                    datum.language = row["language"]
                    datum.cdo_id = row["cdo_id"]
                    datum.deleted = row["deleted"]

                    db.session.merge(datum)
                db.session.commit()
            elif table == "user_contact_text":
                print("Populating table `user_contact_text`.")
                for row in data[table]:
                    datum = UserContactText()
                    datum.id = row["id"]
                    datum.updated = row["updated"]
                    datum.value = row["value"]
                    datum.visible = row["visible"]
                    datum.workplace_user_id = row["workplace_user_id"]
                    datum.contact_type_id = row["contact_type_id"]
                    datum.language = row["language"]
                    datum.cdo_id = row["cdo_id"]
                    datum.deleted = row["deleted"]

                    db.session.merge(datum)
                db.session.commit()
            elif table == "user_contact_photo":
                print("Populating table `user_contact_photo`.")
                for row in data[table]:
                    datum = UserContactPhoto()
                    datum.id = row["id"]
                    datum.updated = row["updated"]
                    datum.value = row["value"]
                    datum.filename = row["filename"]
                    datum.extension = row["extension"]
                    datum.workplace_user_id = row["workplace_user_id"]
                    datum.contact_type_id = row["contact_type_id"]
                    datum.language = row["language"]
                    datum.cdo_id = row["cdo_id"]
                    datum.deleted = row["deleted"]

                    db.session.merge(datum)
                db.session.commit()
            elif table == "workplace_users":
                print("Populating table `workplace_users`.")
                for row in data[table]:
                    datum = WorkplaceUsers()
                    datum.id = row["id"]
                    datum.updated = row["updated"]
                    datum.user_id = row["user_id"]
                    datum.workplace_id = row["workplace_id"]
                    datum.job_title_id = row["job_title_id"]
                    datum.default = row["default"]
                    datum.is_leader = row["is_leader"]
                    datum.cdo_id = row["cdo_id"]
                    datum.deleted = row["deleted"]
                    datum.job_type_id = row["job_type_id"]

                    db.session.merge(datum)
                db.session.commit()
            elif table == "user_workplace_permission":
                print("Populating table `user_workplace_permission`.")
                for row in data[table]:
                    datum = UserWorkplacePermissions()
                    datum.id = row["id"]
                    datum.updated = row["updated"]
                    datum.user_id = row["user_id"]
                    datum.workplace_id = row["workplace_id"]

                    db.session.merge(datum)
                db.session.commit()
            elif table == "job_titles":
                print("Populating table `job_titles`.")
                for row in data[table]:
                    datum = JobTitles()
                    datum.id = row["id"]
                    datum.updated = row["updated"]
                    datum.deleted = row["deleted"]

                    db.session.merge(datum)
                db.session.commit()
            elif table == "job_title_langs":
                print("Populating table `job_title_langs`.")
                for row in data[table]:
                    datum = JobTitleLangs()
                    datum.id = row["id"]
                    datum.updated = row["updated"]
                    datum.job_title_id = row["job_title_id"]
                    datum.language = row["language"]
                    datum.full_job_title = row["full_job_title"]

                    db.session.merge(datum)
                db.session.commit()
            elif table == "workplace_langs":
                print("Populating table `workplace_langs`.")
                for row in data[table]:
                    datum = WorkplaceLangs()
                    datum.id = row["id"]
                    datum.updated = row["updated"]
                    datum.workplace_id = row["workplace_id"]
                    datum.language = row["language"]
                    datum.title = row["title"]
                    datum.abbreviation = row["abbreviation"]

                    db.session.merge(datum)
                db.session.commit()
            elif table == "workplaces":
                print("Populating table `workplaces`.")
                for row in data[table]:
                    datum = Workplaces()
                    datum.id = row["id"]
                    datum.updated = row["updated"]
                    datum.tree_path = row["treePath"]
                    datum.deleted = row["deleted"]

                    db.session.merge(datum)
                db.session.commit()

    update = LastUpdate()
    update.id = 0
    update.time = datetime.now()

    print("Local database populated successfully.")


def getUserData(db: SQLAlchemy, user_login: str) -> dict[str, str | int | dict[str, list[str]]] | None:
    userData = {}
    indices = ["firstName",
               "middleName",
               "lastName",
               "titleBefore",
               "titleAfter",
               "phone",
               "email",
               "room",
               "link",
               "social",
               "note",
               "consultationHours",
               "photo",  # str
               "workplace",  # list[dict[str, list[str]]]
               "publications",
               "candle"]
    for index in indices:
        if index == "firstName" or index == "middleName" or index == "lastName" or index == "titleBefore" or \
                index == "titleAfter" or index == "consultationHours" or index == "photo":
            userData[index] = None
        else:
            userData[index] = []

    result = db.session.query(Users).filter(Users.login == user_login).all()
    if len(result) == 0 or result[0].deleted == 1:
        return None

    userId = result[0].id
    uoc = result[0].uoc
    userData["userId"] = userId

    userInfoResult = db.session.query(UserInfo).filter(UserInfo.user_id == userId).all()
    if len(userInfoResult) == 0:
        return None

    userInfoResult = userInfoResult[0]

    firstName = userInfoResult.first_name
    middleName = userInfoResult.middle_name
    lastName = userInfoResult.last_name
    titleBefore = userInfoResult.title_before
    titleAfter = userInfoResult.title_after

    userData["firstName"] = firstName
    userData["middleName"] = middleName
    userData["lastName"] = lastName
    userData["titleBefore"] = titleBefore
    userData["titleAfter"] = titleAfter

    workplaceUsersResult = db.session.query(WorkplaceUsers).filter(WorkplaceUsers.user_id == userId).all()
    if len(workplaceUsersResult) != 0:
        for rowWUR in workplaceUsersResult:
            workplaceUserId = rowWUR.id
            # default = rowWUR.default
            deleted = rowWUR.deleted

            if deleted == 0:
                userContactVarcharResult = db.session.query(UserContactVarchar).filter(
                    UserContactVarchar.workplace_user_id == workplaceUserId).all()
                if len(userContactVarcharResult) != 0:
                    for rowUCVR in userContactVarcharResult:
                        varcharValue = rowUCVR.value
                        varcharVisible = rowUCVR.visible
                        varcharDeleted = rowUCVR.deleted
                        varcharContactTypeId = rowUCVR.contact_type_id

                        if varcharVisible == 1 and varcharDeleted == 0:
                            contactTypesResult = db.session.query(ContactTypes).filter(
                                ContactTypes.id == varcharContactTypeId).all()
                            if len(contactTypesResult) != 0:
                                contactTypeName = contactTypesResult[0].name

                                if contactTypeName == "phone":
                                    userData["phone"].append(varcharValue)
                                elif contactTypeName == "email":
                                    userData["email"].append(varcharValue)
                                elif contactTypeName == "room":
                                    userData["room"].append(varcharValue)
                                elif contactTypeName == "link":
                                    userData["link"].append(varcharValue)
                                elif contactTypeName == "social":
                                    userData["social"].append(varcharValue)

                userContactTextResult = db.session.query(UserContactText).filter(
                    UserContactText.workplace_user_id == workplaceUserId).all()
                if len(userContactTextResult) != 0:
                    for rowUCTR in userContactTextResult:
                        textValue = rowUCTR.value
                        textVisible = rowUCTR.visible
                        textDeleted = rowUCTR.deleted
                        textContactTypeId = rowUCTR.contact_type_id

                        if textVisible == 1 and textDeleted == 0:
                            contactTypesResult = db.session.query(ContactTypes).filter(
                                ContactTypes.id == textContactTypeId).all()
                            if len(contactTypesResult) != 0:
                                contactTypeName = contactTypesResult[0].name

                                if contactTypeName == "note":
                                    userData["note"].append(textValue)
                                elif contactTypeName == "consultation_hours":
                                    userData["consultationHours"] = textValue

                userData["photo"] = "https://sluzby.fmph.uniba.sk/f/" + user_login

                workplaceId = rowWUR.workplace_id
                workplacesResult = db.session.query(Workplaces).filter(Workplaces.id == workplaceId).all()
                oneWorkplace = {}

                if len(workplacesResult) != 0:
                    rowWP = workplacesResult[0]
                    # workplacesId = rowWP.id
                    workplacesTreePath = rowWP.tree_path
                    workplacesDeleted = rowWP.deleted

                    if workplacesDeleted == 0:
                        workplaces = workplacesTreePath.split(",")
                        uni = workplaces[0]
                        faculty = workplaces[1]

                        workplaceLangsResult = db.session.query(WorkplaceLangs).filter(
                            WorkplaceLangs.workplace_id == uni, WorkplaceLangs.language == "sk_SK").all()
                        oneWorkplace["uni"] = workplaceLangsResult[0].title

                        workplaceLangsResult = db.session.query(WorkplaceLangs).filter(
                            WorkplaceLangs.workplace_id == faculty, WorkplaceLangs.language == "sk_SK").all()
                        if len(workplaceLangsResult) != 0:
                            for rowWPLR in workplaceLangsResult:
                                oneWorkplace["faculty"] = rowWPLR.title
                        else:
                            oneWorkplace["faculty"] = None

                        if len(workplaces) > 2:
                            department = workplaces[2]
                            workplaceLangsResult = db.session.query(WorkplaceLangs).filter(
                                WorkplaceLangs.workplace_id == department, WorkplaceLangs.language == "sk_SK").all()
                            if len(workplaceLangsResult) != 0:
                                for rowWPLR in workplaceLangsResult:
                                    oneWorkplace["department"] = rowWPLR.title
                            else:
                                oneWorkplace["department"] = None

                jobTitleId = rowWUR.job_title_id
                # print("jobTitleId: ", jobTitleId)
                jobTitlesResult = db.session.query(JobTitles).filter(JobTitles.id == jobTitleId).all()
                if jobTitlesResult[0].deleted == 0:
                    jobTitleLangsResult = db.session.query(JobTitleLangs).filter(
                        JobTitleLangs.job_title_id == jobTitleId, JobTitleLangs.language == "sk_SK")

                    for rowJTLR in jobTitleLangsResult:
                        oneWorkplace["positions"] = rowJTLR.full_job_title

                userData["workplace"].append(oneWorkplace)

    publications = getUserPublications(uoc)

    for pub in publications:
        publication = {}
        authors = re.sub(r"\([^)]*%\)", "", pub["autori"])
        publication["authors"] = authors

        name = pub["nazov"]
        publication["name"] = name

        location = pub["kde"]
        publication["location"] = location

        userData["publications"].append(publication)

    candleURL = "https://candle.fmph.uniba.sk/ucitelia/" + user_login + ".csv"
    response = requests.get(candleURL).content.decode()

    reader = csv.reader(response.split('\n'), delimiter=',')
    for i, row in enumerate(reader):
        if len(row) == 0:
            break
        if i == 0:
            continue
        else:
            dayTimeTable = {}
            dayTimeTable["day"] = row[0]
            dayTimeTable["timeFrom"] = row[1]
            dayTimeTable["timeUntil"] = row[2]
            dayTimeTable["duration"] = row[3]
            dayTimeTable["room"] = row[4]
            dayTimeTable["form"] = row[5]
            dayTimeTable["subject"] = row[6]
            dayTimeTable["teacher"] = row[7]
            dayTimeTable["note"] = row[8]

            userData["candle"].append(dayTimeTable)

    return userData
