import java.io.BufferedReader;
import java.io.IOException;
import java.io.InputStreamReader;
import java.math.BigDecimal;
import java.sql.*;
import java.text.SimpleDateFormat;
import java.time.LocalDate;
import java.time.LocalDateTime;
import java.time.format.DateTimeParseException;
import java.util.ArrayList;
import java.util.Arrays;

public class Customer {
    int id;
    String firstName;
    String lastName;
    LocalDate birthDate;
    BigDecimal credits;
    int points;

    public Customer(){
        this.id = -1;                        //pokladna
        credits = new BigDecimal(-1);   //infinity
    }

    //get funkcie
    public int getId(){
        return id;
    }

    public String getFirstName(){
        return firstName;
    }

    public String getLastName(){
        return lastName;
    }

    public LocalDate getBirthDate(){
        return birthDate;
    }

    public BigDecimal getCredits(){
        return credits;
    }

    public int getPoints(){
        return points;
    }

    //set funkcie
    public void setId(int id) {
        this.id = id;
    }

    public void setFirstName(String firstName) {
        this.firstName = firstName;
    }

    public void setLastName(String lastName) {
        this.lastName = lastName;
    }

    public void setBirthDate(LocalDate birthDate) {
        this.birthDate = birthDate;
    }

    public void setCredits(BigDecimal credits) {
        this.credits = credits;
    }

    public void setPoints(int points) {
        this.points = points;
    }

    //ostatne funkcie
    public void insert() throws SQLException {
        String sql = "INSERT INTO zakaznici (meno, priezvisko, datum_narodenia, kredity, body) VALUES (?,?,?,?,?)";
        try(PreparedStatement s = DbContext.getConnection().prepareStatement(sql, Statement.RETURN_GENERATED_KEYS)){
            s.setString(1, firstName);
            s.setString(2, lastName);
            s.setDate(3, Date.valueOf(birthDate));
            s.setBigDecimal(4, credits);
            s.setInt(5, points);

            s.executeUpdate();

            try(ResultSet r = s.getGeneratedKeys()) {
                r.next();
                id = r.getInt("zakaznik_id");
            }
        }
    }

    public void update() throws SQLException {
        String sql = "UPDATE zakaznici SET meno = ?, priezvisko = ?, datum_narodenia = ? WHERE zakaznik_id = ?";
        try(PreparedStatement s = DbContext.getConnection().prepareStatement(sql)){
            s.setString(1, firstName);
            s.setString(2, lastName);
            s.setDate(3, Date.valueOf(birthDate));
            s.setInt(4, id);

            s.executeUpdate();
        }
    }

    public void delete() throws SQLException {
        String sql = "DELETE FROM zakaznici WHERE zakaznik_id = ?";
        try(PreparedStatement s = DbContext.getConnection().prepareStatement(sql)){
            s.setInt(1, id);

            s.executeUpdate();
        }
    }

    //Funkcie na pridavanie kreditu pre uzivatela

    public void updateCredit(BigDecimal amount) throws SQLException {
        String sql = "UPDATE zakaznici SET kredity = ? WHERE zakaznik_id = ?";
        try(PreparedStatement s = DbContext.getConnection().prepareStatement(sql)){
            s.setBigDecimal(1, credits.add(amount));
            s.setInt(2, id);

            s.executeUpdate();
        }
    }

    public void addCredit() throws SQLException, IOException {
        BufferedReader br = new BufferedReader(new InputStreamReader(System.in));

        try {
            System.out.println("Select amount > ");
            BigDecimal amount = new BigDecimal(br.readLine().strip());

            if (amount.compareTo(BigDecimal.ZERO) > 0) {
                updateCredit(amount);
                System.out.println("Credit has been succesfully added");
            } else {
                System.out.println("You cannot add amount less or equal to 0");
            }
        }
        catch(NumberFormatException e){
            System.out.println("Wrong number format for amount. Try again later.");
        }
    }

    //edit

    public void editCustomer() throws SQLException, IOException {
        BufferedReader br = new BufferedReader(new InputStreamReader(System.in));

        try {
            System.out.println("New first name > ");
            setFirstName(br.readLine().strip());
            System.out.println("New last name > ");
            setLastName(br.readLine().strip());
            System.out.println("New birth day (yyyy-MM-dd) > ");
            setBirthDate(LocalDate.parse(br.readLine().strip()));

            update();
        }
        catch(DateTimeParseException e){
            System.out.println("Wrong date format! Try again later.");
        }
    }

    public void showTickets(String state){
        try{
            ArrayList<Ticket> tickets;
            switch (state) {
                case "r":
                    tickets = Ticket.findTicketsByCustomerIdWithState(id, "R");   //rezervovany
                    break;
                case "n":
                    tickets = Ticket.findTicketsByCustomerIdWithState(id, "N");   //nepouzity
                    break;
                case "u":
                    tickets = Ticket.findTicketsByCustomerIdWithState(id, "P");   //pouzity
                    break;
                default:
                    System.out.println("Wrong type. Try again later.");
                    return;
            }

            for (Ticket ticket : tickets) {
                Printer.printInline(ticket);
            }
        }
        catch(SQLException e){
            System.out.println("Something went wrong. Try again later.");
        }
    }

    /**
     * <p>
     *     Funkcia selectTicket() ponuka uzivatelovi plny servis pri nakupe listkov na stadione.
     *     Umoznuje mu miesta rezervovat, vidiet jeho registracie ci zakupit rezervovane listky.
     * </p>
     * @param customerId ID-cko pouzivatela, ktory nakupuje listky
     * @throws SQLException
     * @throws IOException
     */
    public void selectTickets(Event event) throws SQLException, IOException {
        BufferedReader br = new BufferedReader(new InputStreamReader(System.in));
        ArrayList<Ticket> reserved = Ticket.findTicketsByCustomerIdWithState(id, "R");  //R - rezervovane

        System.out.println("You are making a reservation for event: " + event.getName());
        System.out.println("+-----------------------------+");
        System.out.println("What you want to do? Type ...");
        System.out.println("Sector     - if you want to see free seats for event");
        System.out.println("Reserved   - show your reserved tickets");
        System.out.println("Add        - make a reservation");
        System.out.println("Remove     - remove ticket from reservations");
        System.out.println("Pay        - pay your reserved tickets");    //stav N davat kupenym!
        System.out.println("End        - back to menu");

        boolean ordering = true;
        while(ordering){
            System.out.println("What's your option? Select a command > ");
            String option = br.readLine().strip().toLowerCase();
            switch(option){
                case "sector":
                    event.showSector(id);
                    break;
                case "reserved":
                    for(Ticket ticket: reserved){
                        Printer.printInline(ticket);
                    }
                    break;
                case "add":
                    Ticket ticket = addTicketToCart(event);
                    if(ticket != null){
                        reserved.add(ticket);
                        System.out.println("Ticket successfully added to your cart.");
                    }
                    break;
                case "remove":
                    int index = getIndexOfItemInArray(reserved);
                    if(index != -1){
                        reserved.get(index).delete();
                        reserved.remove(index);
                    }
                    else{
                        System.out.println("Wrong index. No ticket was remove.");
                    }
                    break;
                case "pay":
                    payTickets();
                    ordering = false;
                    break;
                case "end":
                    return;
                default:
                    System.out.println("Not a option. Repeat!");
                    break;
            }
        }
    }

    /**
     * <p>
     *     Funkcia zabezpecuje servis pri rezervacii miesta. Zisti od uzivatela,
     *     o ktore miesto ma zaujem a nasledne sa pokusi mu dane miesto na akcii
     *     zarezervovat. V pripade neuspechu vracia null.
     * </p>
     * @param event instancia Event-u
     * @return listok instancia triedy Ticket, inak null
     * @throws IOException
     */
    public Ticket addTicketToCart(Event event) throws IOException {
        BufferedReader br = new BufferedReader(new InputStreamReader(System.in));

        try{
            System.out.println("[D, St, N, Se] - kid, student, normal or senior ticket");
            System.out.println("Select type > ");
            String priceType = br.readLine().strip();

            if(! new ArrayList<>(Arrays.asList("d", "st", "n", "se")).contains(priceType.toLowerCase())){
                System.out.println("You write wrong type. Try again.");
                return null;
            }

            System.out.println("Select sector > ");
            int sectorId = Integer.parseInt(br.readLine().strip());

            System.out.println("Select row > ");
            int row = Integer.parseInt(br.readLine().strip());

            System.out.println("Select place > ");
            int place = Integer.parseInt(br.readLine().strip());

            Place placeInstance = Place.findIdByValues(place, row, sectorId);
            if(placeInstance != null){
                if(placeInstance.isPlaceFree(event.getId())){  //, sectorId, place, row
                    Ticket ticket = new Ticket();

                    ticket.setEventId(event.getId());
                    ticket.setCustomerId(id);
                    ticket.setPlaceId(placeInstance.getId());
                    ticket.setType(priceType);
                    ticket.setState("R");       //cize spravi rezervaciu, preto state = 'R'
                    ticket.setPrice(PriceList.findById(event.getPriceListId()).getPriceByType(priceType));
                    ticket.setTime(Timestamp.valueOf(LocalDateTime.now()));

                    ticket.insert();
                    return ticket;
                }
                else {
                    System.out.println("This place is not available.");
                    return null;
                }
            }
            else{
                System.out.println("This place doesn't exist.");
                return null;
            }
        }
        catch(NumberFormatException e){
            System.out.println("You don't write a correct number. Cannot be parsed.");
            return null;
        }
        catch(SQLException e){
            System.out.println("Something went wrong. Try Again.");
            return null;
        }
    }

    /**
     * Funkcia sluzi na zistenie indexu istka v poli cart, ktory chce uzivatel zrusit
     * @param cart pole listkov
     * @return index index listku v poli cart, ktory chce pouzivatel zrusit
     * @throws SQLException
     * @throws IOException
     */
    public int getIndexOfItemInArray(ArrayList<Ticket> cart) throws SQLException, IOException {
        BufferedReader br = new BufferedReader(new InputStreamReader(System.in));

        int id = 1;
        for(Ticket ticket: cart){
            System.out.print(id + ". | ");
            Printer.printInline(ticket);
            id++;
        }

        try{
            System.out.println("Select ticket's ID > ");
            int index = Integer.parseInt(br.readLine().strip()) - 1;

            if(index >= 0 && index < cart.size()){
                return index;
            }
            else {
                System.out.println("Number out of range. Try again.");
                return -1;
            }
        }
        catch(NumberFormatException e){
            System.out.println("You don't write a correct number. Try again later.");
            return -1;
        }
    }

    public void payTickets() throws SQLException, IOException {
        payTickets(false);      //nie je to z pokladne
    }

    public void payTickets(boolean itIsFromMainMenu) throws SQLException, IOException {
        BufferedReader br = new BufferedReader(new InputStreamReader(System.in));
        ArrayList<Ticket> reserved = Ticket.findTicketsByCustomerIdWithState(id, "R");
        ArrayList<Ticket> cart = new ArrayList<>();

        System.out.println("You are paying your reserved tickets.");
        System.out.println("+-----------------------------+");
        System.out.println("What you want to do? Type ...");
        System.out.println("Reserved   - show your reserved tickets");
        System.out.println("Cart       - show tickets in your cart");
        System.out.println("Add        - add ticket to your cart");
        System.out.println("Remove     - remove ticket from your cart");
        System.out.println("Pay        - buy tickets in your cart");    //stav N davat kupenym!
        System.out.println("End        - back to menu");

        for(Ticket ticket: cart){
            Printer.printInline(ticket);
        }


        boolean paying = true;
        int index = -1;
        BigDecimal price;
        while(paying){
            System.out.println("What's your option? Select a command > ");
            String option = br.readLine().strip().toLowerCase();

            switch(option){
                case "reserved":
                    for(Ticket ticket: reserved){
                        Printer.printInline(ticket);
                    }
                    break;
                case "cart":
                    price = new BigDecimal(0);
                    for(Ticket ticket: cart){
                        Printer.printInline(ticket);
                        price = price.add(ticket.getPrice());
                    }
                    System.out.println("-- Total price: " + price);
                    break;
                case "add":
                    index = getIndexOfItemInArray(reserved);
                    if(index != -1) {
                        cart.add(reserved.remove(index));       //zoberie ticket z rezervovanych a prida ich do kosika
                    }
                    else{
                        System.out.println("Wrong index. No ticket was added.");
                    }
                    break;
                case "remove":
                    index = getIndexOfItemInArray(cart);
                    if(index != -1) {
                        reserved.add(cart.remove(index));       //zoberie ticket z kosika a prida ich naspat do rezervovanych
                    }
                    else{
                        System.out.println("Wrong index. No ticket was remove.");
                    }
                    break;
                case "pay":
                    price = new BigDecimal(0);
                    for(Ticket ticket: cart) price = price.add(ticket.getPrice());

                    if(itIsFromMainMenu){       //ak plati z pokladne rezervovane listky pre uzivatela
                        if(-1 == credits.intValue()){       //z pokladne aj rezervoval a teraz plati pre uzivatela s id == null
                            credits = new BigDecimal(price.intValue() + 1000);      //istota
                        }
                        updateCredit(price);    //inac len pridam cenu listkov, ved to zaplati priamo na pokladni
                    }
                    if(1 == credits.compareTo(price)){
                        pay(cart);
                        sendCredits(1, price);
                        System.out.println("Tickets are successfuly payed. Total price = " + price);
                    }
                    else {
                        System.out.println("You have not enough credits.");
                        System.out.println("You want to pay for " + cart.size() + " ticket, total price = " + price);
                        System.out.println("You have only " + credits);
                    }
                    break;
                case "end":
                    paying = false;
                    break;
                default:
                    System.out.println("Not a option. Repeat!");
                    break;
            }
        }

    }

    public void useTicket() throws SQLException {
        try {
            BufferedReader br = new BufferedReader(new InputStreamReader(System.in));
            ArrayList<Ticket> nonUsedTickets = Ticket.findTicketsByCustomerIdWithState(id, "N");

            System.out.println("Which ticket do you want to mark as used?");
            System.out.println("Here is list of yours unused tickets:");
            System.out.println("+-----------------------------+");
            int index = getIndexOfItemInArray(nonUsedTickets);
            if(index != -1) {
                Ticket ticket = nonUsedTickets.get(index);
                SimpleDateFormat format = new SimpleDateFormat("yyyy-MM-dd");
                Timestamp eventDate = Event.findById(ticket.getEventId()).getEventDate();
                Timestamp todayDate = new Timestamp(System.currentTimeMillis());


                if(format.format(eventDate).equals(format.format(todayDate))){
                    DbContext.getConnection().setTransactionIsolation(Connection.TRANSACTION_SERIALIZABLE);
                    DbContext.getConnection().setAutoCommit(false);

                    //nastavi stav na 'P' ako pouzity a updatne ho
                    ticket.setState("P");
                    ticket.updateState();

                    System.out.println("Ticket successfuly mark as 'used'.");
                }
                else{
                    System.out.println("You can mark ticket as 'used' only on date of event.");
                    System.out.println("Today is: " + todayDate);
                    System.out.println("Date of event: " + eventDate);
                }
            }
        }
        catch(SQLException | IOException e){
            DbContext.getConnection().rollback();
            System.out.println("Something went wrong. Try again later.");
        }
        finally {
            DbContext.getConnection().setAutoCommit(true);
        }
    }

    //hladacie funkcie
    public static Customer makeCustomerFromResultSet(ResultSet r) throws SQLException {
        Customer customer = new Customer();

        customer.setId(r.getInt("zakaznik_id"));
        customer.setFirstName(r.getString("meno"));
        customer.setLastName(r.getString("priezvisko"));
        customer.setBirthDate(r.getDate("datum_narodenia").toLocalDate());
        customer.setCredits(r.getBigDecimal("kredity"));
        customer.setPoints(r.getInt("body"));

        return customer;
    }

    public void pay(ArrayList<Ticket> cart) throws SQLException{
        DbContext.getConnection().setTransactionIsolation(Connection.TRANSACTION_SERIALIZABLE);
        DbContext.getConnection().setAutoCommit(false);

        try{
            for(Ticket ticket: cart){
                ticket.setState("N");  //N - kupeny, nepouzity
                ticket.updateState();
            }
        }
        catch(SQLException e){
            DbContext.getConnection().rollback();
            throw e;
        }
        finally {
            DbContext.getConnection().setAutoCommit(true);
        }
    }

    public void sendCredits(int recipientId, BigDecimal price) throws SQLException {
        DbContext.getConnection().setTransactionIsolation(Connection.TRANSACTION_SERIALIZABLE);
        DbContext.getConnection().setAutoCommit(false);

        try{
            Customer recipient = Customer.findById(recipientId);

            //mne odcitaj peniaze a recipientovi pridaj
            updateCredit(price.negate());   //mne
            recipient.updateCredit(price);  //recipientovi

            //nastavenie bodov za nakup (debilny sposob, ale funguje)
            double points =  Double.valueOf(String.valueOf(price.divide(BigDecimal.valueOf(5))));
            setPoints(getPoints() + (int)points);
            update();
        }
        catch(SQLException e){
            DbContext.getConnection().rollback();
            throw e;
        }
        finally {
            DbContext.getConnection().setAutoCommit(true);
        }
    }

    public static Customer findById(int customerId) throws SQLException {
        String sql = "SELECT * FROM zakaznici WHERE zakaznik_id = ?";
        try(PreparedStatement s = DbContext.getConnection().prepareStatement(sql)){
            s.setInt(1, customerId);

            try(ResultSet r = s.executeQuery()){
                if(r.next()){
                    Customer customer = makeCustomerFromResultSet(r);

                    if(r.next()){
                        throw new RuntimeException("More than one customer with id = " + customerId);
                    }

                    return customer;
                }
                return null;    //customer so zadanym ID nenajdeny
            }
        }
    }

    public static ArrayList<Customer> findAll() throws SQLException {
        String sql = "SELECT * FROM zakaznici";
        try(PreparedStatement s = DbContext.getConnection().prepareStatement(sql)){
            try(ResultSet r = s.executeQuery()){
                ArrayList<Customer> result = new ArrayList<>();

                while(r.next()){
                    result.add(makeCustomerFromResultSet(r));
                }

                return result;
            }
        }
    }
}
