import javax.print.DocFlavor;
import java.io.BufferedReader;
import java.io.IOException;
import java.io.InputStreamReader;
import java.sql.*;
import java.time.LocalDate;
import java.time.LocalDateTime;
import java.util.ArrayList;
import java.util.Arrays;


/**
 * <p>
 *     Trieda Event slúži na reprezentáciu Akcie.
 *     Tak isto osahuje metódy, ktoré slúžia na prácu s akciami.
 * </p>
 */
public class Event {
    int id;
    String name;
    Timestamp eventDate;
    String type;
    String state = "N";     //U - koncene, N - ne-ukoncene (defaultne N, nakolko akcia nemoze mat skorsi datum ako 'now()')
    int priceListId;
    ArrayList<Integer> sectors;

    //get funkcie

    /**
     * Vracia {@code id}
     * @return id akcie
     */
    public int getId() { return id; }

    /**
     * Vracia {@code name}
     * @return meno akcie
     */
    public String getName() { return name; }

    /**
     * Vracia {@code eventDate}
     * @return dátum akcie
     */
    public Timestamp getEventDate() { return eventDate; }

    /**
     * Vracia {@code type}
     * @return typ akcie
     */
    public String getType() { return type; }

    /**
     * Vracia {@code state}
     * @return stav akcie
     */
    public String getState() { return state; }

    /**
     * Vracia {@code priceListId}
     * @return ID-čko cenníka
     */
    public int getPriceListId() { return priceListId; }

    /**
     * Vracia
     * @return
     * @throws SQLException
     */
    public ArrayList<Integer> getSectors() throws SQLException {
        String sql = "SELECT * FROM priradene_sektory WHERE akcia_id = ?";
        try(PreparedStatement s = DbContext.getConnection().prepareStatement(sql)){
            s.setInt(1, id);

            try(ResultSet r = s.executeQuery()){
                ArrayList<Integer> result = new ArrayList<>();

                while(r.next()){
                    result.add(Integer.valueOf(r.getInt("sektor_id")));
                }

                return result;
            }
        }
    }

    //set funkcie
    public void setId(int id) { this.id = id; }
    public void setName(String name) { this.name = name; }
    public void setEventDate(Timestamp eventDate) { this.eventDate = eventDate; }
    public void setType(String type) { this.type = type; }
    public void setState(String state) { this.state = state; }
    public void setPriceListId(int priceListId) { this.priceListId = priceListId; }
    public void setSectors(ArrayList<Integer> sectors) { this.sectors = sectors; }

    //ostatne funkcie

    /**
     * Funkcia vkladá novú akciu do tabuľky {@code akcie}
     * @throws SQLException
     */
    public void insert() throws SQLException {
        String sql = "INSERT INTO akcie (nazov, datum, typ, stav, cennik_id) VALUES (?, ?, ?, ?, ?)";
        try(PreparedStatement s = DbContext.getConnection().prepareStatement(sql, Statement.RETURN_GENERATED_KEYS)){
            s.setString(1, name);
            s.setTimestamp(2, eventDate);
            s.setString(3, type);
            s.setString(4, state);
            s.setInt(5, priceListId);

            s.executeUpdate();

            try(ResultSet r = s.getGeneratedKeys()) {
                r.next();
                id = r.getInt("akcia_id");
            }
        }
        catch(SQLException e){
            DbContext.getConnection().rollback();
            throw e;
        }
    }

    /**
     * Funkcia update-u záznam v tabuľke {@code akcie}
     * @throws SQLException
     */
    public void update() throws SQLException {
        String sql = "UPDATE akcie SET nazov = ?, datum = ?, typ = ?, stav = ?, cennik_id = ? WHERE akcia_id = ?";
        DbContext.getConnection().setTransactionIsolation(Connection.TRANSACTION_SERIALIZABLE);
        DbContext.getConnection().setAutoCommit(false);
        try(PreparedStatement s = DbContext.getConnection().prepareStatement(sql)){
            s.setString(1, name);
            s.setTimestamp(2, eventDate);
            s.setString(3, type);
            s.setString(4, state);
            s.setInt(5, priceListId);
            s.setInt(6, id);

            s.executeUpdate();
        }
        catch(SQLException e){
            DbContext.getConnection().rollback();
            throw e;
        }
        finally {
            DbContext.getConnection().setAutoCommit(true);
        }
    }

    /**
     * Funkcia zmaže záznam z tabuľky {@code akcie}
     * @throws SQLException
     */
    public void delete() throws SQLException {
        String sql = "DELETE FROM akcie WHERE akcia_id = ?";
        try(PreparedStatement s = DbContext.getConnection().prepareStatement(sql)){
            s.setInt(1, id);

            s.executeUpdate();
        }
    }

    /**
     * Funkcia slúži hlavne pri vypisovaní informácii o akcii.
     * Na základe typu akcie vráti plný názov typu tejto akcie.
     * @return plný názov typu akcie
     */
    public String getTypeFullName(){
        switch(type){
            case "G": return "Regular game";
            case "P": return "Play-off game";
            case "E": return "Special event";
            default: return "Unknown type";
        }
    }

    /**
     * <p>
     *     Funkcia vkladá do tabuľky {@code priradene_miesta} záznami
     *     o tom, ktoré sektory sú pre danú akciu otvorené.
     * </p>
     * @throws SQLException
     */
    public void setSectorsOnStadium() throws SQLException {
        String sql = "INSERT INTO priradene_sektory (akcia_id, sektor_id) VALUES (?, ?)";
        for(Integer sektorId: sectors){
            try(PreparedStatement s = DbContext.getConnection().prepareStatement(sql)){
                s.setInt(1, id);
                s.setInt(2, sektorId);

                s.executeUpdate();
            }
        }
    }

    /**
     *<p>
     *     Funkciazoberie textový vstup od použivateľa
     *     a snaží sa z neho vytvoriť pole obsahujúce ID-čka sektorov, ktoré uživateľ
     *     chce mať na danej akcii otvorené.
     *</p>
     * @param sectorsInString Textový vstup od použivateľa popisujúci sektory, ktoré chce priradiť akcii
     * @param maximumSectorId Počet sektorov na štadióne
     * @return pole Integerov obsahujúce ID-čka sektorov, ktoré sú otvorené na akcii
     * @throws IllegalArgumentException
     */
    public ArrayList<Integer> makeSectorsFromString(String sectorsInString, int maximumSectorId){
        ArrayList<Integer> sectorIds = new ArrayList<>();
        String[] sectors = sectorsInString.split(",");

        for(String sector: sectors){
            if(sector.contains("-")){
                String[] tmp = sector.split("-");
                String from = tmp[0]; String to = tmp[1];
                if(Integer.parseInt(to) > maximumSectorId){
                    throw new IllegalArgumentException("Sector id is bigger than maximum sector id.");
                }
                for(int id = Integer.parseInt(from); id <= Integer.parseInt(to); id++){
                    sectorIds.add(id);
                }
            }
            else{
                sectorIds.add(Integer.parseInt(sector));
            }
        }

        this.setSectors(sectorIds);
        return sectorIds;
    }

    /**
     * Funkcia spočíta počet miest v hľadisku pre danú akciu.
     * @return počet miest v hľadisku
     * @throws SQLException
     */
    public int totalPlacesCounter() throws SQLException {
        int counter = 0;
        for(Integer sektor_id: getSectors()) {
            String sql = "SELECT count(*) FROM miesta WHERE sektor_id = ?";
            try(PreparedStatement s = DbContext.getConnection().prepareStatement(sql)){
                s.setInt(1, sektor_id);

                try(ResultSet r = s.executeQuery()){
                    while(r.next()) {
                        counter += r.getInt("count");
                    }
                }
            }
        }

        return counter;
    }

    /**
     * Funkcia spocita počet predaných lístkov na danú akciu.
     * @return počet predaných lístkov
     * @throws SQLException
     */
    public int sellTicketsCounter() throws SQLException {
        String sql = "SELECT count(*) FROM listky WHERE akcia_id = ?";
        try(PreparedStatement s = DbContext.getConnection().prepareStatement(sql)){
            s.setInt(1, id);

            try(ResultSet r = s.executeQuery()){
                r.next();
                return r.getInt("count");
            }
        }
    }

    public int totalPrice() throws SQLException {
        String sql = "SELECT count(cena) AS total_cena FROM listky WHERE akcia_id = ?";
        try(PreparedStatement s = DbContext.getConnection().prepareStatement(sql)){
            s.setInt(1, id);

            try(ResultSet r = s.executeQuery()){
                r.next();
                return r.getInt("total_cena");
            }
        }
    }

    /**
     * Funkcia zistuje, ci je sektor na danu akciu otvoreny a mozu sa nan predavat listky
     * @param sectorId ID-cko sektora
     * @return true, ak je dany sektor pre akciu otvoreny
     * @throws SQLException
     */
    public boolean canBuyTicketInSector(int sectorId) throws SQLException {
        return getSectors().contains(Integer.valueOf(sectorId));
    }

    public void showSector() throws SQLException, IOException {
        showSector(-1);
    }

    public void showSector(int customerId) throws IOException {
        BufferedReader br = new BufferedReader(new InputStreamReader(System.in));

        System.out.println("Enter sector's id > ");

        try {
            int sectorId = Integer.parseInt(br.readLine().strip());

            if(canBuyTicketInSector(sectorId)) {
                Printer.printSector(id, sectorId, customerId);
            }
            else{
                System.out.println("Sector with id = " + sectorId + " is not open for this event.");
            }
        }
        catch(NumberFormatException e){
            System.out.println("You don't write a correct number. Cannot be parsed.");
        }
        catch(SQLException e){
            System.out.println("Something went wrong. Try Again.");
        }
    }

    //hladacie funkcie

    /**
     * Funkcia zoberie ResultSet a vytvori z neho instanciu triedy 'Event'
     * @param r ResultSet
     * @return event Event vytvoreny z ResultSet-u
     * @throws SQLException
     */
    public static Event makeEventFromResultSet(ResultSet r) throws SQLException {
        Event event = new Event();

        event.setId(r.getInt("akcia_id"));
        event.setName(r.getString("nazov"));
        event.setEventDate(r.getTimestamp("datum"));
        event.setType(r.getString("typ"));
        event.setState(r.getString("stav"));
        event.setPriceListId(r.getInt("cennik_id"));

        return event;
    }

    /**
     * Funkcia vrati {@code count} akcii, zoradene od dnes
     * @param count Pocet akcii, ktore chce pouzivatel zobrazit
     * @return result pole obsahujuce Event-y
     * @throws SQLException
     */
    public static ArrayList<Event> findEventOrderByTime(int count) throws SQLException{
        String sql = "SELECT * FROM akcie WHERE datum >= ? ORDER BY datum ASC LIMIT ?";
        try(PreparedStatement s = DbContext.getConnection().prepareStatement(sql)){
            s.setTimestamp(1, Timestamp.valueOf(LocalDateTime.now()));
            s.setInt(2, count);

            try(ResultSet r = s.executeQuery()){
                ArrayList<Event> result = new ArrayList<>();

                while(r.next()){
                    result.add(makeEventFromResultSet(r));
                }

                return result;
            }
        }
    }

    /**
     * <p>
     *      Funkcia podla zadaneho {@code eventId} v tabulke 'akcie' najde a nasledne vytvori
     *      instanciu triedy Event. Ak akciu nenajde, vracia null.
     * </p>
     * @param eventId
     * @return event instancia triedy Event, inak null
     * @throws SQLException
     */
    public static Event findById(int eventId) throws SQLException {
        String sql = "SELECT * FROM akcie WHERE akcia_id = ?";
        try(PreparedStatement s = DbContext.getConnection().prepareStatement(sql)){
            s.setInt(1, eventId);

            try(ResultSet r = s.executeQuery()){
                if(r.next()){
                    Event event = makeEventFromResultSet(r);

                    if(r.next()){
                        throw new RuntimeException("More than one event with id = " + eventId);
                    }

                    return event;
                }
                return null;    //event so zadanym id nenajdeny
            }
        }
    }

    /**
     * Funkcia najde a vrati vsetky akcie z tabulky 'akcie'
     * @return result Pole obsahujuce vsetky akcie
     * @throws SQLException
     */
    public static ArrayList<Event> findAll() throws SQLException {
        String sql = "SELECT * FROM akcie";
        try(PreparedStatement s = DbContext.getConnection().prepareStatement(sql)){
            try(ResultSet r = s.executeQuery()){
                ArrayList<Event> result = new ArrayList<>();

                while(r.next()){
                    result.add(makeEventFromResultSet(r));
                }

                return result;
            }
        }
    }

}
