<?php

namespace App\Command;

use App\Entity\MailNotification;
use App\Repository\UserRepository;
use App\Services\EveningCheckService;
use App\Services\Mailer;
use DateTimeImmutable;
use Doctrine\ORM\EntityManagerInterface;
use Doctrine\ORM\NonUniqueResultException;
use Exception;
use Symfony\Component\Console\Attribute\AsCommand;
use Symfony\Component\Console\Command\Command;
use Symfony\Component\Console\Input\InputInterface;
use Symfony\Component\Console\Output\OutputInterface;
use Symfony\Component\Console\Style\SymfonyStyle;
use Symfony\Contracts\Translation\TranslatorInterface;

#[AsCommand(
    name: 'app:check-evening-checks',
    description: 'Check if all active students with room have evening check attendance',
)]
class CheckEveningChecksCommand extends Command
{
    private EveningCheckService $eveningCheckService;
    private Mailer $mailer;
    private UserRepository $userRepository;
    private EntityManagerInterface $entityManager;
    private TranslatorInterface $translator;

    public function __construct(EveningCheckService $eveningCheckService, Mailer $mailer,
                                UserRepository $userRepository, EntityManagerInterface $entityManager, TranslatorInterface $translator)
    {
        parent::__construct();
        $this->eveningCheckService = $eveningCheckService;
        $this->mailer = $mailer;
        $this->userRepository = $userRepository;
        $this->entityManager = $entityManager;
        $this->translator = $translator;
    }

    /**
     * @throws NonUniqueResultException
     * @throws Exception
     */
    protected function execute(InputInterface $input, OutputInterface $output): int
    {
        $io = new SymfonyStyle($input, $output);
        $io->progressStart();
        $date = new DateTimeImmutable();
        if ($date->format('H') >= 23) {
            // send incomplete evening check, educator had plenty of time to record evening attendance
            $this->saveMailNotification($io, $date, MailNotification::INCOMPLETE_EVENING_CHECK_TYPE);
        }
        else if ($this->eveningCheckService->checkAllStudentsHaveEveningCheck()){
            $this->saveMailNotification($io, $date, MailNotification::COMPLETE_EVENING_CHECK_TYPE);
            $io->success('All students have completed their evening check!');
        } else {
            $io->error('Not all students have completed their evening check yet.');
        }

        return Command::SUCCESS;
    }
    private function saveMailNotification(SymfonyStyle $io, DateTimeImmutable $date, int $type): void
    {
        $users = $this->userRepository->findAllActiveEducatorsAndAdmins();
        $io->progressAdvance();
        if (empty($users)) {
            $io->error('did not find users to send email');
            return;
        }
        foreach ($users as $user) {
            $io->progressAdvance();
            $this->mailer->saveMailNotification($user, $this->translator->trans('entity.mail_notification.evening_check_done'), $date, $type);
        }
        $this->entityManager->flush();
        $io->progressFinish();
    }
}
