<?php

namespace App\Command;

use App\Entity\Meeting;
use App\Repository\MeetingRepository;
use App\Services\NotificationService;
use App\Services\PushNotifier;
use App\Services\UserSubscriptionManager;
use DateTimeImmutable;
use Doctrine\ORM\EntityManagerInterface;
use EasyCorp\Bundle\EasyAdminBundle\Router\AdminUrlGenerator;
use Exception;
use Symfony\Component\Console\Attribute\AsCommand;
use Symfony\Component\Console\Command\Command;
use Symfony\Component\Console\Input\InputInterface;
use Symfony\Component\Console\Output\OutputInterface;
use Symfony\Component\Console\Style\SymfonyStyle;
use Symfony\Contracts\Translation\TranslatorInterface;

#[AsCommand(
    name: 'app:check-today-meeting',
    description: 'Checking if educator or students have today meeting',
)]
class CheckForTodayMeetingCommand extends Command
{
    private PushNotifier $pushNotifier;
    private EntityManagerInterface $entityManager;
    private UserSubscriptionManager $userSubscriptionManager;
    private MeetingRepository $meetingRepository;
    private AdminUrlGenerator $adminUrlGenerator;
    private TranslatorInterface $translator;

    public function __construct(PushNotifier $pushNotifier, EntityManagerInterface $entityManager, UserSubscriptionManager $userSubscriptionManager,
                                MeetingRepository $meetingRepository, AdminUrlGenerator $adminUrlGenerator, TranslatorInterface $translator)
    {
        parent::__construct();
        $this->pushNotifier = $pushNotifier;
        $this->entityManager = $entityManager;
        $this->userSubscriptionManager = $userSubscriptionManager;
        $this->meetingRepository = $meetingRepository;
        $this->adminUrlGenerator = $adminUrlGenerator;
        $this->translator = $translator;
    }

    /**
     * @throws Exception
     */
    protected function execute(InputInterface $input, OutputInterface $output): int
    {
        $io = new SymfonyStyle($input, $output);
        $io->progressStart();
        $date = new DateTimeImmutable();
        $todayMeetings = $this->meetingRepository->findTodayMeetings($date);
        $io->progressAdvance();
        /** @var Meeting $meeting */
        foreach ($todayMeetings as $meeting) {
            $io->progressAdvance();
            $notificationService = new NotificationService($this->userSubscriptionManager, $this->adminUrlGenerator, $this->pushNotifier, $this->translator);
            $notificationService->createNotificationForMeeting($meeting, true); // educator
            $notificationService->createNotificationForMeeting($meeting, false); // students
        }
        $io->progressAdvance();
        $this->entityManager->flush();
        $io->progressFinish();
        $io->success('Check for today meeting done.');

        return Command::SUCCESS;
    }

}
