<?php

namespace App\Command;

use App\Entity\Activity;
use App\Entity\Attendance;
use App\Entity\Student;
use App\Repository\ActivityRepository;
use App\Repository\AttendanceRepository;
use App\Repository\StudentRepository;
use App\Services\NotificationService;
use App\Services\PushNotifier;
use App\Services\UserSubscriptionManager;
use DateTimeImmutable;
use Doctrine\ORM\EntityManagerInterface;
use Doctrine\ORM\NonUniqueResultException;
use EasyCorp\Bundle\EasyAdminBundle\Router\AdminUrlGenerator;
use Exception;
use Symfony\Component\Console\Attribute\AsCommand;
use Symfony\Component\Console\Command\Command;
use Symfony\Component\Console\Input\InputInterface;
use Symfony\Component\Console\Output\OutputInterface;
use Symfony\Component\Console\Style\SymfonyStyle;
use Symfony\Contracts\Translation\TranslatorInterface;

#[AsCommand(
    name: 'app:check-if-returned',
    description: 'Checking if students did not return from walk, training, departure, or any other attendance',
)]
class CheckIfStudentsReturnedBackCommand extends Command
{
    private PushNotifier $pushNotifier;
    private EntityManagerInterface $entityManager;
    private AttendanceRepository $attendanceRepository;
    private UserSubscriptionManager $userSubscriptionManager;
    private StudentRepository $studentRepository;
    private ActivityRepository $activityRepository;
    private AdminUrlGenerator $adminUrlGenerator;
    private TranslatorInterface $translator;

    public function __construct(PushNotifier $pushNotifier, EntityManagerInterface $entityManager, UserSubscriptionManager $userSubscriptionManager,
                                AttendanceRepository $attendanceRepository, StudentRepository $studentRepository,
                                ActivityRepository $activityRepository, AdminUrlGenerator $adminUrlGenerator, TranslatorInterface $translator)
    {
        parent::__construct();
        $this->pushNotifier = $pushNotifier;
        $this->entityManager = $entityManager;
        $this->attendanceRepository = $attendanceRepository;
        $this->userSubscriptionManager = $userSubscriptionManager;
        $this->studentRepository = $studentRepository;
        $this->activityRepository = $activityRepository;
        $this->adminUrlGenerator = $adminUrlGenerator;
        $this->translator = $translator;
    }

    /**
     * @throws Exception
     */
    protected function execute(InputInterface $input, OutputInterface $output): int
    {
        $io = new SymfonyStyle($input, $output);
        $io->progressStart();
        $date = new DateTimeImmutable();
        $activeStudents = $this->studentRepository->findActiveStudentsWithRoom();
        $otherAttendances = $this->attendanceRepository->findByStudentsArrayAndDate($activeStudents, $date);
        $io->progressAdvance();
        /** @var Attendance $otherAttendance */
        // for now - walk, departure, training
        $notificationService = new NotificationService($this->userSubscriptionManager, $this->adminUrlGenerator, $this->pushNotifier, $this->translator);
        foreach ($otherAttendances as $otherAttendance) {
            $io->progressAdvance();
            if ($otherAttendance->getValidUntil() < $date) {
                $student = $otherAttendance->getStudent();
                if($this->lookIfNotReturned($student, $otherAttendance->getValidUntil())){
                    $notificationService->createNotificationStudentNotReturned($student, $otherAttendance->getActionToText());
                }
            }
        }
        $trainings = $this->activityRepository->findTrainingsByStudentsAndDay($activeStudents, $date);
        /** @var Activity $training */
        foreach ($trainings as $training) {
            $io->progressAdvance();
            $trainingStudents = $training->getStudents();
            foreach ($trainingStudents as $trainingStudent) {
                $newDateTime = new DateTimeImmutable($date->format('Y-m-d') . ' ' . $training->getToTime()->format('H:i:s'));
                if ($newDateTime < $date) {
                    if($this->lookIfNotReturned($trainingStudent, $newDateTime)){
                        $notificationService->createNotificationStudentNotReturned($trainingStudent, $training->getTypeToText());
                    }
                }
            }
        }
        $this->entityManager->flush();
        $io->progressFinish();
        $io->success('Check for student not returned done.');

        return Command::SUCCESS;
    }

    /**
     * @throws NonUniqueResultException
     */
    public function lookIfNotReturned(?Student $student, DateTimeImmutable $time): bool
    {
        /** @var Attendance $presentOrMissingAttendance */
        // check if at validUntil time we have absent or present attendance
        $presentOrMissingAttendance = $this->attendanceRepository->findAbsentOrPresentForTime($student, $time);
        if ($presentOrMissingAttendance !== null && $presentOrMissingAttendance->getAction() === Attendance::ABSENT) {
            // if absent until validTime to return, check if after validTime we have presentAction
            $presentAction = $this->attendanceRepository->findPresentActionToFuture($student, $time);
            if ($presentAction === null) {
                // if not present action, student did not return, send notification
                return true;
            }
        }
        return false;
    }
}
