<?php

namespace App\Command;

use App\Entity\Meeting;
use App\Entity\User;
use App\Repository\MeetingRepository;
use App\Repository\UserRepository;
use App\Services\NotificationService;
use App\Services\PushNotifier;
use App\Services\UserSubscriptionManager;
use DateTimeImmutable;
use Doctrine\ORM\EntityManagerInterface;
use EasyCorp\Bundle\EasyAdminBundle\Router\AdminUrlGenerator;
use Exception;
use Symfony\Component\Console\Attribute\AsCommand;
use Symfony\Component\Console\Command\Command;
use Symfony\Component\Console\Input\InputArgument;
use Symfony\Component\Console\Input\InputInterface;
use Symfony\Component\Console\Output\OutputInterface;
use Symfony\Component\Console\Style\SymfonyStyle;
use Symfony\Component\Validator\Constraints\Email;
use Symfony\Component\Validator\Constraints\Regex;
use Symfony\Component\Validator\Validator\ValidatorInterface;
use Symfony\Contracts\Translation\TranslatorInterface;
use function PHPUnit\Framework\isEmpty;

#[AsCommand(
    name: 'app:create-admin',
    description: 'Creates a new admin',
)]
class CreateAdminCommand extends Command
{

    private UserRepository $userRepository;
    private EntityManagerInterface $entityManager;
    private ValidatorInterface $validator;

    public function __construct(EntityManagerInterface $entityManager, UserRepository $userRepository, ValidatorInterface $validator)
    {
        parent::__construct();
        $this->userRepository = $userRepository;
        $this->entityManager = $entityManager;
        $this->validator = $validator;
    }
    protected function configure(): void
    {
        $this
            ->setHelp('This command allows you to create an admin')
            ->addArgument('email', InputArgument::REQUIRED, 'The email of the new admin.')
            ->addArgument('firstName', InputArgument::REQUIRED, 'The first name of the new admin.')
            ->addArgument('lastName', InputArgument::REQUIRED, 'The last name of the new admin.')

        ;
    }

    /**
     * @throws Exception
     */
    protected function execute(InputInterface $input, OutputInterface $output): int
    {
        $io = new SymfonyStyle($input, $output);
        $io->progressStart();
        $email = $input->getArgument('email');
        $firstName = $input->getArgument('firstName');
        $lastName = $input->getArgument('lastName');
        $errors = $this->validator->validate($email, new Email());
        $io->progressAdvance();
        if ($errors->count() !== 0){
            $io->error('Invalid email address');
            return Command::SUCCESS;
        }
        $errors = $this->validator->validate($firstName, new Regex('/^[a-zA-Z]+$/'));
        $io->progressAdvance();
        if ($errors->count() !== 0){
            $io->error('First name should contain letters a-z and A-Z');
            return Command::SUCCESS;
        }
        $errors = $this->validator->validate($lastName, new Regex('/^[a-zA-Z]+$/'));
        $io->progressAdvance();
        if ($errors->count() !== 0){
            $io->error('Last name should contain letters a-z and A-Z');
            return Command::SUCCESS;
        }
        $io->progressAdvance();
        $user = $this->userRepository->findOneBy(['email' => $email]);
        if ($user !== null) {
            // user already exists, do not say if existed
            $io->progressFinish();
            $io->success('Admin created');
            return Command::SUCCESS;
        }
        $admin = new User();
        $admin->setActive(true);
        $admin->setRoles([User::ROLE_ADMIN]);
        $admin->setEmail($email);
        $admin->setFirstName($firstName);
        $admin->setLastName($lastName);
        $errors = $this->validator->validate($admin);
        if ($errors->count() !== 0){
            $io->error('Invalid email, last name or first name.');
            return Command::SUCCESS;
        }
        $this->entityManager->persist($admin);
        $this->entityManager->flush();
        $io->success('Admin created');
        $io->progressFinish();
        return Command::SUCCESS;
    }

}
