<?php

namespace App\Controller;

use App\Entity\Attendance;
use App\Entity\DormitoryCard;
use App\Entity\Student;
use App\Repository\DormitoryCardRepository;
use App\Repository\StudentRepository;
use DateTimeImmutable;
use Doctrine\ORM\EntityManagerInterface;
use Doctrine\ORM\NonUniqueResultException;
use Symfony\Bundle\FrameworkBundle\Controller\AbstractController;
use Symfony\Component\HttpFoundation\JsonResponse;
use Symfony\Component\HttpFoundation\Request;
use Symfony\Component\HttpFoundation\Response;
use Symfony\Component\Routing\Annotation\Route;

#[Route('/system/scan-card', name: 'app_system_scan_card_')]
class APICardController extends AbstractController
{
    private EntityManagerInterface $entityManager;

    public function __construct(EntityManagerInterface $entityManager) {

        $this->entityManager = $entityManager;
    }

    #[Route('/create', name: 'create', methods: ['GET'])]
    public function create(string $raspberry_api_key, DormitoryCardRepository $dormitoryCardRepository, Request $request): Response
    {
        $api_key = $request->get('api_key');
        if ($api_key === null || $api_key !== $raspberry_api_key){
            return new Response(null, 404);
        }
        $card = $dormitoryCardRepository->findOneBy(['number' => $request->get('rfid_id')]);
        if (null === $card) {
            //create card in db
            $card = new DormitoryCard();
            $card->setNumber($request->get('rfid_id'));
            $card->setStudent(null);
            $card->setCreationTime(new DateTimeImmutable());
            $this->entityManager->persist($card);
            $this->entityManager->flush();
            return new Response('card created');
        }
        return new Response('card already exist', 404);
    }

    /**
     * @throws NonUniqueResultException
     */
    #[Route('/attendance', name: 'attendance', methods: ['GET'])]
    public function arrived(string $raspberry_api_key, Request $request, StudentRepository $studentRepository): Response
    {
        $api_key = $request->get('api_key');
        if ($api_key === null || $api_key !== $raspberry_api_key){
            return new Response(null, 404);
        }
        $type = null;
        if ($request->get('type') === 'ARRIVE'){
            $type = true;
        }
        elseif ($request->get('type') === 'DEPARTURE'){
            $type = false;
        }
        if ($type === null){
            return new Response('Wrong type', 404);
        }
        $student = $studentRepository->findActiveStudentWithRoomAndDormitoryCard($request->get('rfid_id'));
        if (null !== $student) {
            $this->logAttendance($student, $type);
            return new Response('attendance logged');
        }
        return new Response('student not found', 400);
    }

    #[Route('/send_data', name: 'send_data', methods: ['GET'])]
    public function sendData(string $raspberry_api_key, Request $request, DormitoryCardRepository $dormitoryCardRepository): JsonResponse
    {
        $api_key = $request->get('api_key');
        if ($api_key === null || $api_key !== $raspberry_api_key){
            return new JsonResponse(null, 404);
        }
        $activeCards = $dormitoryCardRepository->findActiveCards();
        return new JsonResponse($activeCards);
    }

    private function logAttendance(Student $student, bool $type): void
    {
        $attendance = new Attendance();
        $attendance->setStudent($student);
        $attendance->setTime(new DateTimeImmutable());
        if ($type) {
            // true - set arrived
            $attendance->setAction(Attendance::PRESENT);
        }
        else {
            // false - set departed
            $attendance->setAction(Attendance::ABSENT);
        }
        $this->entityManager->persist($attendance);
        $this->entityManager->flush();
    }
}
