<?php

namespace App\Controller\Admin;

use App\Controller\Admin\Crud\ClassroomCrudController;
use App\Controller\Admin\Crud\CoachCrudController;
use App\Controller\Admin\Crud\EducatorCrudController;
use App\Controller\Admin\Crud\GuardianCrudController;
use App\Controller\Admin\Crud\LeavingHomeCrudController;
use App\Controller\Admin\Crud\MasterTeacherCrudController;
use App\Controller\Admin\Crud\MeetingCrudController;
use App\Controller\Admin\Crud\RoomCrudController;
use App\Controller\Admin\Crud\SchoolCrudController;
use App\Controller\Admin\Crud\StudentCrudController;
use App\Controller\Admin\Crud\StudentNoteCrudController;
use App\Controller\Admin\Crud\WalkOutCrudController;
use App\Entity\Attendance;
use App\Entity\Classroom;
use App\Entity\Coach;
use App\Entity\Educator;
use App\Entity\Guardian;
use App\Entity\MasterTeacher;
use App\Entity\Meeting;
use App\Entity\Room;
use App\Entity\School;
use App\Entity\Student;
use App\Entity\StudentNote;
use App\Entity\Activity;
use App\Entity\User;
use App\Repository\MessageReceiverRepository;
use App\Repository\ScheduleRepository;
use App\Services\ScheduleService;
use DateTimeImmutable;
use EasyCorp\Bundle\EasyAdminBundle\Config\Action;
use EasyCorp\Bundle\EasyAdminBundle\Config\Actions;
use EasyCorp\Bundle\EasyAdminBundle\Config\Assets;
use EasyCorp\Bundle\EasyAdminBundle\Config\Crud;
use EasyCorp\Bundle\EasyAdminBundle\Config\Dashboard;
use EasyCorp\Bundle\EasyAdminBundle\Config\MenuItem;
use EasyCorp\Bundle\EasyAdminBundle\Config\UserMenu;
use EasyCorp\Bundle\EasyAdminBundle\Controller\AbstractDashboardController;
use EasyCorp\Bundle\EasyAdminBundle\Router\AdminUrlGenerator;
use Exception;
use Symfony\Component\HttpFoundation\Response;
use Symfony\Component\Routing\Annotation\Route;
use Symfony\Component\Security\Core\User\UserInterface;
use Symfony\Component\Security\Http\Attribute\IsGranted;
use Symfony\Contracts\Translation\TranslatorInterface;

#[IsGranted('ROLE_ADMIN')]
class AdminDashboardController extends AbstractDashboardController
{

    private ScheduleRepository $scheduleRepository;
    private AdminUrlGenerator $adminUrlGenerator;
    private MessageReceiverRepository $messageReceiverRepository;
    private TranslatorInterface $translator;

    public function __construct(ScheduleRepository $scheduleRepository, AdminUrlGenerator $adminUrlGenerator,
                                MessageReceiverRepository $messageReceiverRepository, TranslatorInterface $translator)
    {
        $this->scheduleRepository = $scheduleRepository;
        $this->adminUrlGenerator = $adminUrlGenerator;
        $this->messageReceiverRepository = $messageReceiverRepository;
        $this->translator = $translator;
    }

    /**
     * @throws Exception
     */
    #[Route('/admin', name: 'admin')]
    public function index(): Response
    {
        $date = new DateTimeImmutable();
        $educatorUrl = $this->generateCrudUrl(EducatorCrudController::class);
        $studentUrl = $this->generateCrudUrl(StudentCrudController::class);
        $meetingUrl = $this->generateCrudUrl(MeetingCrudController::class);
        $walkUrl = $this->generateCrudUrl(WalkOutCrudController::class);
        $leaveUrl = $this->generateCrudUrl(LeavingHomeCrudController::class);
        $noteUrl = $this->generateCrudUrl(StudentNoteCrudController::class);
        $roomUrl = $this->generateCrudUrl(RoomCrudController::class);
        $messageUrl =  $this->adminUrlGenerator
            ->setDashboard(AdminDashboardController::class)
            ->setRoute('app_message_show_received')
            ->generateUrl();
        $attendanceUrl = $this->adminUrlGenerator
            ->setDashboard(AdminDashboardController::class)
            ->setRoute('app_attendance_index')
            ->generateUrl();
        return $this->render('dashboard.html.twig', [
            'schedules' => $this->scheduleRepository->findScheduleForDate($date),
            'currentDate' => $date,
            'freeDay' => (new ScheduleService)->checkIfFreeDay($date),
            'unreadMessagesCount' => $this->messageReceiverRepository->count(['isRead' => false, 'receiver' => $this->getUser()]),
            'url' => [
                'educator' => $educatorUrl,
                'student' => $studentUrl,
                'meeting' => $meetingUrl,
                'walk' => $walkUrl,
                'leave' => $leaveUrl,
                'note' => $noteUrl,
                'message' => $messageUrl,
                'room' => $roomUrl,
                'attendance' => $attendanceUrl,
            ]
        ]);
    }

    public function configureDashboard(): Dashboard
    {
        return Dashboard::new()
            ->setTitle($this->translator->trans('dashboard.main_name'));
    }
    public function configureCrud(): Crud
    {
        return Crud::new()
            // the max number of entities to display per page
            ->setPaginatorPageSize(30)
            // the number of pages to display on each side of the current page
            // e.g. if num pages = 35, current page = 7 and you set ->setPaginatorRangeSize(4)
            // the paginator displays: [Previous]  1 ... 3  4  5  6  [7]  8  9  10  11 ... 35  [Next]
            // set this number to 0 to display a simple "< Previous | Next >" pager
            ->setPaginatorRangeSize(4)
            ;
    }
    public function configureAssets(): Assets
    {
        return parent::configureAssets()
            ->addWebpackEncoreEntry('easyadmin');
    }

    public function configureActions(): Actions
    {
        return parent::configureActions()
            ->add(Crud::PAGE_INDEX, Action::DETAIL)
            ->update(Crud::PAGE_DETAIL, Action::EDIT, static function (Action $action) {
                return $action->setIcon('fa fa-edit');
            })
            ->update(Crud::PAGE_DETAIL, Action::INDEX, static function (Action $action) {
                return $action->setIcon('fa fa-list');
            })
            ;
    }

    public function configureMenuItems(): iterable
    {
        yield MenuItem::linkToDashboard('dashboard.main_panel', 'fa fa-home');
        yield MenuItem::linkToRoute('entity.message.plural_name', 'fas fa-envelope', 'app_message_show_received');
        yield MenuItem::linkToRoute('entity.schedule.singular_name', 'fas fa-calendar-days', 'app_schedule_index');

        yield MenuItem::linkToCrud('entity.educator.plural_name', 'fas fa-user-tie', Educator::class)
            ->setQueryParameter('filters[activeUser]', 1)
            ->setController(EducatorCrudController::class);

        yield MenuItem::section('');
        yield MenuItem::linkToCrud('entity.meeting.plural_name', 'fas fa-handshake-simple', Meeting::class)
            ->setController(MeetingCrudController::class);
        yield MenuItem::linkToCrud('entity.attendance.action.walk_out.plural_name', 'fas fa-shoe-prints', Attendance::class)
            ->setController(WalkOutCrudController::class);
        yield MenuItem::linkToCrud('entity.attendance.action.departure.plural_name', 'fas fa-paper-plane', Attendance::class)
            ->setController(LeavingHomeCrudController::class);
        yield MenuItem::linkToRoute('entity.attendance.singular_name', 'fas fa-clock', 'app_attendance_index');

        yield MenuItem::section('');
        yield MenuItem::subMenu('dashboard.accommodated_students', 'fas fa-users')
            ->setSubItems([
                MenuItem::linkToCrud('entity.student.plural_name', 'fas fa-user-graduate', Student::class)
                    ->setQueryParameter('filters[activeUser]', 1)
                    ->setController(StudentCrudController::class),
                MenuItem::linkToRoute('entity.dormitory_card.plural_name', 'fas fa-id-card', 'app_dormitory_card_show'),
                MenuItem::linkToCrud('entity.guardian.plural_name', 'fas fa-hands-holding-child', Guardian::class)
                    ->setQueryParameter('filters[activeUser]', 1)
                    ->setController(GuardianCrudController::class),
                MenuItem::linkToCrud('entity.student_note.plural_name', 'fas fa-edit', StudentNote::class)
                    ->setController(StudentNoteCrudController::class),

            ]);
        yield MenuItem::linkToCrud('entity.room.plural_name', 'fas fa-bed', Room::class)
            ->setController(RoomCrudController::class);
        yield MenuItem::linkToCrud('entity.coach.plural_name', 'fas fa-person-running', Coach::class)
            ->setQueryParameter('filters[activeUser]', 1)
            ->setController(CoachCrudController::class);
        yield MenuItem::linkToCrud('entity.activity.type.training.plural_name', 'fas fa-basketball', Activity::class);
        yield MenuItem::linkToCrud('entity.master_teacher.plural_name', 'fas fa-chalkboard-user', MasterTeacher::class)
            ->setQueryParameter('filters[activeUser]', 1)
            ->setController(MasterTeacherCrudController::class);
        yield MenuItem::linkToCrud('entity.school.plural_name', 'fas fa-school', School::class)
            ->setController(SchoolCrudController::class);
        yield MenuItem::linkToCrud('entity.classroom.plural_name', 'fas fa-chalkboard', Classroom::class)
                    ->setController(ClassroomCrudController::class);

    }
    public function configureUserMenu(UserInterface $user): UserMenu
    {
        /** @var User $user */
        return parent::configureUserMenu($user)
            ->setName($user->getFirstName() . ' ' . $user->getLastName() . ' ' . $user->getEmail())
            ->addMenuItems([
                MenuItem::linkToRoute('entity.user.profile', 'fa fa-id-card', 'app_user_profile'),
            ]);
    }

    private function generateCrudUrl(string $class): string
    {
        return $this->adminUrlGenerator
            ->setDashboard(AdminDashboardController::class)
            ->setController($class)
            ->setAction('index')
            ->generateUrl();
    }
}
