<?php

namespace App\Controller\Admin\Crud;

use App\Controller\Admin\Filter\ActiveUserFilter;
use App\Entity\Educator;
use App\Entity\User;
use Doctrine\ORM\QueryBuilder;
use EasyCorp\Bundle\EasyAdminBundle\Collection\FieldCollection;
use EasyCorp\Bundle\EasyAdminBundle\Collection\FilterCollection;
use EasyCorp\Bundle\EasyAdminBundle\Config\Action;
use EasyCorp\Bundle\EasyAdminBundle\Config\Actions;
use EasyCorp\Bundle\EasyAdminBundle\Config\Crud;
use EasyCorp\Bundle\EasyAdminBundle\Config\Filters;
use EasyCorp\Bundle\EasyAdminBundle\Controller\AbstractCrudController;
use EasyCorp\Bundle\EasyAdminBundle\Dto\EntityDto;
use EasyCorp\Bundle\EasyAdminBundle\Dto\SearchDto;
use EasyCorp\Bundle\EasyAdminBundle\Field\AssociationField;
use EasyCorp\Bundle\EasyAdminBundle\Field\ChoiceField;
use EasyCorp\Bundle\EasyAdminBundle\Field\CountryField;
use EasyCorp\Bundle\EasyAdminBundle\Field\EmailField;
use EasyCorp\Bundle\EasyAdminBundle\Field\IdField;
use EasyCorp\Bundle\EasyAdminBundle\Field\TelephoneField;
use EasyCorp\Bundle\EasyAdminBundle\Field\TextField;
use Symfony\Component\Security\Http\Attribute\IsGranted;

#[IsGranted(User::ROLE_ADMIN)]
class EducatorCrudController extends AbstractCrudController
{
    public static function getEntityFqcn(): string
    {
        return Educator::class;
    }
    public function createEntity(string $entityFqcn): Educator
    {
        $educator = new Educator();
        $user = new User();
        $user->setRoles([User::ROLE_EDUCATOR]);
        $educator->setUser($user);

        return $educator;
    }
    public function createIndexQueryBuilder(SearchDto $searchDto, EntityDto $entityDto, FieldCollection $fields, FilterCollection $filters): QueryBuilder
    {
        return parent::createIndexQueryBuilder($searchDto, $entityDto, $fields, $filters)
            ->addSelect('user_index')
            ->leftJoin('entity.user', 'user_index');
    }
    public function configureCrud(Crud $crud): Crud
    {
        return $crud
            ->setEntityLabelInSingular('entity.educator.singular_name')
            ->setEntityLabelInPlural('entity.educator.plural_name')
            ->setDefaultSort(['id' => 'ASC',])
            ->setSearchFields(['user.firstName', 'user.lastName', 'user.email', 'user.phoneNumber'])
            ;
    }

    public function configureFilters(Filters $filters): Filters
    {
        return $filters
            ->add(ActiveUserFilter::new('activeUser', 'entity.user.active.name'))
            ;
    }
    public function configureActions(Actions $actions): Actions
    {
        return $actions
            ->remove(Crud::PAGE_DETAIL, Action::DELETE)
            ->remove(Crud::PAGE_INDEX, Action::DELETE)
            ->disable(Action::BATCH_DELETE, Action::DELETE)
            ;
    }

    public function configureFields(string $pageName): iterable
    {
        return [
            IdField::new('id')
                ->setLabel('entity.id')
                ->hideOnForm(),
            TextField::new('user.firstName')
                ->setLabel('entity.user.first_name')
                ->hideOnForm(),
            TextField::new('user.lastName')
                ->setLabel('entity.user.last_name')
                ->hideOnForm(),
            EmailField::new('user.email')
                ->setLabel('entity.user.email')
                ->hideOnForm(),
            ChoiceField::new('user.active')
                ->setLabel('entity.user.active.name')
                ->hideOnForm()
                ->setChoices([
                    'entity.user.active.no' => 0,
                    'entity.user.active.yes' => 1]),
            CountryField::new('user.phoneCountry')
                ->setLabel('entity.user.phone_country')
                ->hideOnForm(),
            TelephoneField::new('user.phoneNumber')
                ->setLabel('entity.user.phone_number')
                ->hideOnForm(),
            AssociationField::new('user')
                ->onlyOnForms()
                ->renderAsEmbeddedForm()
                ->setColumns('col-12')
                ->setFormTypeOptions([
                    'label' => false,
                ]),
        ];
    }

}
