<?php

namespace App\Controller\Admin\Crud;

use App\Controller\Admin\Filter\ActiveUserFilter;
use App\Entity\Guardian;
use App\Entity\User;
use Doctrine\ORM\QueryBuilder;
use EasyCorp\Bundle\EasyAdminBundle\Collection\FieldCollection;
use EasyCorp\Bundle\EasyAdminBundle\Collection\FilterCollection;
use EasyCorp\Bundle\EasyAdminBundle\Config\Crud;
use EasyCorp\Bundle\EasyAdminBundle\Config\Filters;
use EasyCorp\Bundle\EasyAdminBundle\Controller\AbstractCrudController;
use EasyCorp\Bundle\EasyAdminBundle\Dto\EntityDto;
use EasyCorp\Bundle\EasyAdminBundle\Dto\SearchDto;
use EasyCorp\Bundle\EasyAdminBundle\Field\AssociationField;
use EasyCorp\Bundle\EasyAdminBundle\Field\ChoiceField;
use EasyCorp\Bundle\EasyAdminBundle\Field\CountryField;
use EasyCorp\Bundle\EasyAdminBundle\Field\EmailField;
use EasyCorp\Bundle\EasyAdminBundle\Field\IdField;
use EasyCorp\Bundle\EasyAdminBundle\Field\TelephoneField;
use EasyCorp\Bundle\EasyAdminBundle\Field\TextField;
use Symfony\Component\Security\Http\Attribute\IsGranted;

#[IsGranted(User::ROLE_ADMIN)]
class GuardianCrudController extends AbstractCrudController
{
    public static function getEntityFqcn(): string
    {
        return Guardian::class;
    }
    public function createEntity(string $entityFqcn): Guardian
    {
        $mother = new Guardian();
        $user = new User();
        $user->setRoles([User::ROLE_GUARDIAN]);
        $mother->setUser($user);

        return $mother;
    }
    public function createIndexQueryBuilder(SearchDto $searchDto, EntityDto $entityDto, FieldCollection $fields, FilterCollection $filters): QueryBuilder
    {
        return parent::createIndexQueryBuilder($searchDto, $entityDto, $fields, $filters)
            ->addSelect('user_index')
            ->leftJoin('entity.user', 'user_index');
    }

    public function configureCrud(Crud $crud): Crud
    {
        return $crud
            ->setEntityLabelInSingular('entity.guardian.singular_name')
            ->setEntityLabelInPlural('entity.guardian.plural_name')
            ->setDefaultSort(['id' => 'ASC',])
            ->setSearchFields(['user.firstName', 'user.lastName', 'user.email', 'user.phoneNumber'])
            ;
    }
    public function configureFilters(Filters $filters): Filters
    {
        return $filters
            ->add(ActiveUserFilter::new('activeUser', 'entity.user.active.name'))
            ->add('children')
            ;
    }
    public function configureFields(string $pageName): iterable
    {
        return [
            IdField::new('id')
                ->setLabel('entity.id')
                ->hideOnForm(),
            AssociationField::new('user')
                ->renderAsEmbeddedForm()
                ->onlyOnForms()
                ->setColumns('col-12')
                ->setFormTypeOptions([
                    'label' => false,
                ]),
            TextField::new('user.firstName')
                ->setLabel('entity.user.first_name')
                ->hideOnForm(),
            TextField::new('user.lastName')
                ->setLabel('entity.user.last_name')
                ->hideOnForm(),
            EmailField::new('user.email')
                ->setLabel('entity.user.email')
                ->hideOnForm(),
            CountryField::new('user.phoneCountry')
                ->setLabel('entity.user.phone_country')
                ->setColumns('col-6')
                ->hideOnIndex()
                ->hideOnForm()
                ->setRequired(false),
            TelephoneField::new('user.phoneNumber')
                ->setColumns('col-6')
                ->hideOnForm()
                ->setLabel('entity.user.phone_number'),
            ChoiceField::new('user.active')
                ->setLabel('entity.user.active.name')
                ->setColumns('col-12')
                ->hideOnForm()
                ->setChoices([
                    'entity.user.active.no' => 0,
                    'entity.user.active.yes' => 1]),
            AssociationField::new('children')
                ->setLabel('entity.guardian.child.plural_name')
                ->autocomplete()
                ->hideWhenCreating()
                ->hideOnIndex()
                ->setColumns('col-12')
                ->setFormTypeOptions([
                    'by_reference' => false,
                ])
                ->setTemplatePath('admin/show_students.html.twig'),
        ];
    }
}
