<?php

namespace App\Controller\Admin\Crud;

use App\Controller\Admin\Filter\MeetingTypeFilter;
use App\Controller\Admin\Filter\StudentMeetingFilter;
use App\Controller\Educator\Crud\MeetingCrudEducatorController;
use App\Entity\Meeting;
use App\Entity\MeetingAttendance;
use App\Entity\User;
use App\Form\Meeting\IndividualMeetingType;
use App\Form\Meeting\MeetingType;
use App\Form\Meeting\NonIndividualMeetingType;
use App\Repository\MeetingAttendanceRepository;
use DateTimeImmutable;
use Doctrine\ORM\EntityManagerInterface;
use Doctrine\ORM\QueryBuilder;
use EasyCorp\Bundle\EasyAdminBundle\Collection\FieldCollection;
use EasyCorp\Bundle\EasyAdminBundle\Collection\FilterCollection;
use EasyCorp\Bundle\EasyAdminBundle\Config\Action;
use EasyCorp\Bundle\EasyAdminBundle\Config\Actions;
use EasyCorp\Bundle\EasyAdminBundle\Config\Crud;
use EasyCorp\Bundle\EasyAdminBundle\Config\Filters;
use EasyCorp\Bundle\EasyAdminBundle\Context\AdminContext;
use EasyCorp\Bundle\EasyAdminBundle\Controller\AbstractCrudController;
use EasyCorp\Bundle\EasyAdminBundle\Dto\EntityDto;
use EasyCorp\Bundle\EasyAdminBundle\Dto\SearchDto;
use EasyCorp\Bundle\EasyAdminBundle\Field\AssociationField;
use EasyCorp\Bundle\EasyAdminBundle\Field\CollectionField;
use EasyCorp\Bundle\EasyAdminBundle\Field\DateTimeField;
use EasyCorp\Bundle\EasyAdminBundle\Field\FormField;
use EasyCorp\Bundle\EasyAdminBundle\Field\IdField;
use EasyCorp\Bundle\EasyAdminBundle\Field\IntegerField;
use EasyCorp\Bundle\EasyAdminBundle\Field\TextField;
use EasyCorp\Bundle\EasyAdminBundle\Router\AdminUrlGenerator;
use Exception;
use Symfony\Component\ExpressionLanguage\Expression;
use Symfony\Component\HttpFoundation\Request;
use Symfony\Component\HttpFoundation\Response;
use Symfony\Component\Security\Http\Attribute\IsGranted;
use Symfony\Component\Translation\TranslatableMessage;
use Symfony\Contracts\Translation\TranslatorInterface;

#[IsGranted(User::ROLE_ADMIN)]
class MeetingCrudController extends AbstractCrudController
{
    private AdminUrlGenerator $adminUrlGenerator;
    protected MeetingAttendanceRepository $meetingAttendanceRepository;
    protected EntityManagerInterface $entityManager;
    private TranslatorInterface $translator;

    public function __construct(AdminUrlGenerator $adminUrlGenerator, EntityManagerInterface $entityManager, TranslatorInterface $translator,
                                MeetingAttendanceRepository $meetingAttendanceRepository)
    {
        $this->adminUrlGenerator = $adminUrlGenerator;
        $this->meetingAttendanceRepository = $meetingAttendanceRepository;
        $this->entityManager = $entityManager;
        $this->translator = $translator;
    }
    public static function getEntityFqcn(): string
    {
        return Meeting::class;
    }
    public function createIndexQueryBuilder(SearchDto $searchDto, EntityDto $entityDto, FieldCollection $fields, FilterCollection $filters): QueryBuilder
    {
        return parent::createIndexQueryBuilder($searchDto, $entityDto, $fields, $filters)
            ->addSelect('meetingAttendances_index')
            ->leftJoin('entity.meetingAttendances', 'meetingAttendances_index')
            ;
    }
    public function configureCrud(Crud $crud): Crud
    {
        return $crud
            ->setEntityLabelInSingular('entity.meeting.singular_name')
            ->setEntityLabelInPlural('entity.meeting.plural_name')
            ->setDefaultSort(['plannedDate' => 'DESC',])
            ->setSearchFields(['educator.user.firstName', 'educator.user.lastName', 'educator.user.email'])
            ->setPageTitle(Crud::PAGE_DETAIL, 'entity.meeting.singular_name')
            ;
    }
    public function configureFilters(Filters $filters): Filters
    {
        return $filters
            ->add('educator')
            ->add('plannedDate')
            ->add(StudentMeetingFilter::new('student', 'entity.student.singular_name'))
            ->add(MeetingTypeFilter::new('type', 'entity.meeting.type'))
            ;
    }
    public function configureActions(Actions $actions): Actions
    {
        $loadMeetingAttendanceAction = Action::new('attendance','entity.meeting.write_attendance')
            ->linkToCrudAction('attendance')
            ->displayIf(static function (Meeting $meeting) {
                $currentDate = new DateTimeImmutable();
                return $meeting->getPlannedDate()->format('Y-m-d') === $currentDate->format('Y-m-d');
            });

        $importIndividualMeetings = Action::new('importIndividualMeetings', 'entity.meeting.import_individual')
            ->linkToRoute('app_import_individual_meeting')
            ->createAsGlobalAction()
            ->setCssClass('btn btn-primary');
        $importGroupMeetings = Action::new('importGroupMeetings', 'entity.meeting.import_group')
            ->linkToRoute('app_import_group_meeting')
            ->createAsGlobalAction()
            ->setCssClass('btn btn-primary');

        $newIndividualMeeting = Action::new('newIndividualMeeting', 'entity.meeting.create_individual')
            ->linkToCrudAction('newIndividual')
            ->createAsGlobalAction()
            ->setCssClass('btn btn-primary');
        $newGroupMeeting = Action::new('newGroupMeeting', 'entity.meeting.create_group')
            ->linkToCrudAction('newGroup')
            ->createAsGlobalAction()
            ->setCssClass('btn btn-primary');
        $editMeeting = Action::new('editMeeting','entity.meeting.edit')
            ->linkToCrudAction('editMeeting')
            ->displayIf(static function (Meeting $meeting) {
                $currentDate = new DateTimeImmutable();
                return $meeting->getPlannedDate()->format('Y-m-d') >= $currentDate->format('Y-m-d');
            });

        $expression = new Expression('is_granted("ROLE_ADMIN") or is_granted("ROLE_EDUCATOR")');
        if (in_array(User::ROLE_EDUCATOR, $this->getUser()->getRoles())){
            $actions->add(Crud::PAGE_INDEX, $loadMeetingAttendanceAction)
                ->setPermissions([
                    'attendance' => User::ROLE_EDUCATOR
                ]);
        }
        return $actions
            ->remove(Crud::PAGE_INDEX, Action::NEW)
            ->remove(Crud::PAGE_INDEX,Action::EDIT)
            ->remove(Crud::PAGE_DETAIL, Action::EDIT)
            ->add(Crud::PAGE_INDEX, $importIndividualMeetings)
            ->add(Crud::PAGE_INDEX, $importGroupMeetings)
            ->add(Crud::PAGE_INDEX, $newIndividualMeeting)
            ->add(Crud::PAGE_INDEX, $newGroupMeeting)
            ->add(Crud::PAGE_INDEX, $editMeeting)
            ->disable(Action::EDIT, Action::NEW)
            ->setPermissions([
                'importIndividualMeetings' => $expression,
                'newIndividualMeeting' => $expression,
                'newGroupMeeting' => $expression,
                'editMeeting' => $expression,
                'importGroupMeetings' => $expression,
            ])
            ;
    }

    public function configureFields(string $pageName): iterable
    {
        return [
            FormField::addTab('dashboard.general_info'),
            IdField::new('id')
                ->setLabel('entity.id')
                ->hideOnForm(),
            AssociationField::new('educator')
                ->setLabel('entity.educator.singular_name')
                ->setCrudController(EducatorCrudController::class)
                ->setColumns('col-12')
                ->autocomplete()
                ->setRequired(true),
            DateTimeField::new('plannedDate')
                ->setLabel('entity.meeting.planned_date')
                ->setRequired(true),
            TextField::new('educatorNote')
                ->setLabel('entity.meeting.educator_note')
                ->setColumns('col-12')
                ->hideOnIndex(),
            TextField::new('studentNote')
                ->setLabel('entity.meeting.student_note')
                ->setColumns('col-12')
                ->hideOnIndex(),
            IntegerField::new('doorNumber')
                ->setLabel('entity.meeting.door_number')
                ->setColumns('col-12')
                ->setRequired(true),
            TextField::new('isPresent')
                ->setLabel('entity.meeting.is_present')
                ->setFormTypeOption('mapped', false)
                ->hideOnForm()
                ->setValue('0')
                ->formatValue(function ($value, $entity) {
                    return $this->countPresentStudentsByMeetingId($entity);
                }),

            TextField::new('studentsCount')
                ->setLabel('entity.meeting.count')
                ->setFormTypeOption('mapped', false)
                ->hideOnForm()
                ->setValue('0')
                ->formatValue(function ($value, $entity) {
                    return $this->countAllStudentsByMeetingId($entity);
                }),
            FormField::addTab('entity.student.plural_name'),
            CollectionField::new('meetingAttendances')
                ->setLabel('entity.meeting.attendance')
                ->hideOnIndex()
                ->setTemplatePath('admin/show_meeting_students.html.twig'),
        ];
    }

    /**
     * @throws Exception
     */
    public function attendance(AdminContext $adminContext, Request $request): Response
    {

        $meeting = $adminContext->getEntity()->getInstance();
        if (!$meeting instanceof Meeting) {
            throw new \LogicException('Entity is missing or not a Meeting');
        }
        $currentDate = new DateTimeImmutable();
        if ($meeting->getPlannedDate()->format('Y-m-d') !== $currentDate->format('Y-m-d')) {
            throw new Exception($this->translator->trans('entity.meeting.invalid_date'));
        }
        $form = $this->createForm(MeetingType::class, $meeting);
        $form->handleRequest($request);
        if ($form->isSubmitted() && $form->isValid()) {
            $this->entityManager->flush();

            $controller = in_array('ROLE_ADMIN', $this->getUser()->getRoles()) ? MeetingCrudController::class : MeetingCrudEducatorController::class;
            /** @var User $user */
            $user = $this->getUser();
            $url = $this->adminUrlGenerator
                ->setDashboard($user->getDashboard())
                ->setController($controller)
                ->setAction(Action::INDEX)
                ->generateUrl();

            return $this->redirect($url);
        }
        return $this->render('admin/load_meeting_attendance.html.twig', [
            'form' => $form,
        ]);
    }

    /**
     * @throws Exception
     */
    public function newIndividual(AdminContext $adminContext, Request $request): Response
    {
        $meeting = new Meeting();
        $educator = $this->getUser()->getEducator();
        $meeting->setEducator($educator);
        $form = $this->createForm(IndividualMeetingType::class,$meeting);
        $form->handleRequest($request);
        if ($form->isSubmitted() && $form->isValid()) {
            $student = $form->get('student')->getData();

            $meetingAttendance = new MeetingAttendance();
            $meetingAttendance->setStudent($student);
            $meeting->addMeetingAttendance($meetingAttendance);
            $this->entityManager->persist($meeting);
            $this->entityManager->persist($meetingAttendance);
            $this->entityManager->flush();
            $this->addFlash("success", new TranslatableMessage('entity.meeting.flash_create_individual', [
                '%student%' => $student,
            ]));
            $controller = in_array('ROLE_ADMIN', $this->getUser()->getRoles()) ? MeetingCrudController::class : MeetingCrudEducatorController::class;
            /** @var User $user */
            $user = $this->getUser();
            $url = $this->adminUrlGenerator
                ->setDashboard($user->getDashboard())
                ->setController($controller)
                ->setAction(Crud::PAGE_INDEX)
                ->generateUrl();
            return $this->redirect($url);
        }
        return $this->render('basic_form.html.twig', [
            'form' => $form,
            'templateUrl' => null,
        ]);
    }

    /**
     * @throws Exception
     */
    public function newGroup(AdminContext $adminContext, Request $request): Response
    {
        $meeting = new Meeting();
        $educator = $this->getUser()->getEducator();
        $meeting->setEducator($educator);
        $form = $this->createForm(NonIndividualMeetingType::class,$meeting);
        $form->handleRequest($request);
        if ($form->isSubmitted() && $form->isValid()) {
            $educator = $form->get('educator')->getData();
            foreach ($educator->getStudents() as $student) {
                if ($student->getUser()->isActive() === true) {
                    $meetingAttendance = new MeetingAttendance();
                    $meetingAttendance->setStudent($student);
                    $meeting->addMeetingAttendance($meetingAttendance);
                    $this->entityManager->persist($meetingAttendance);
                }
            }

            $this->entityManager->persist($meeting);
            $this->entityManager->flush();
            $this->addFlash("success", 'entity.meeting.flash_create_group');
            $controller = in_array('ROLE_ADMIN', $this->getUser()->getRoles()) ? MeetingCrudController::class : MeetingCrudEducatorController::class;
            /** @var User $user */
            $user = $this->getUser();
            $url = $this->adminUrlGenerator
                ->setDashboard($user->getDashboard())
                ->setController($controller)
                ->setAction(Crud::PAGE_INDEX)
                ->generateUrl();
            return $this->redirect($url);
        }
        return $this->render('basic_form.html.twig', [
            'form' => $form,
            'templateUrl' => null,
        ]);
    }

    /**
     * @throws Exception
     */
    public function editMeeting (AdminContext $adminContext, Request $request): Response
    {
        $meeting = $adminContext->getEntity()->getInstance();
        if (!$meeting instanceof Meeting) {
            throw new \LogicException('Entity is missing or not a Meeting');
        }
        $form = $this->createForm(NonIndividualMeetingType::class, $meeting);
        $form->handleRequest($request);
        if ($form->isSubmitted() && $form->isValid()) {
            $this->entityManager->flush();
            $this->addFlash("success", 'entity.meeting.flash_edit');
            $controller = in_array('ROLE_ADMIN', $this->getUser()->getRoles()) ? MeetingCrudController::class : MeetingCrudEducatorController::class;
            /** @var User $user */
            $user = $this->getUser();
            $url = $this->adminUrlGenerator
                ->setDashboard($user->getDashboard())
                ->setController($controller)
                ->setAction(Crud::PAGE_INDEX)
                ->generateUrl();
            return $this->redirect($url);
        }
        return $this->render('basic_form.html.twig', [
            'form' => $form,
            'templateUrl' => null,
        ]);
    }

    protected function countPresentStudentsByMeetingId($entity): int
    {
        /** @var MeetingAttendance $meetingAttendance */
        $count = 0;
        foreach ($entity->getMeetingAttendances() as $meetingAttendance) {
            if ($meetingAttendance->isPresent()) {
                $count++;
            }
        }
        return $count;
    }
    protected function countAllStudentsByMeetingId($entity): int
    {
        return $entity->getMeetingAttendances()->count();
    }
    protected function individualOrGroupMeeting($entity): string
    {
        if ($entity->getMeetingAttendances()->count() == 1) {
            return $this->translator->trans('entity.meeting.individual');
        }
        return $this->translator->trans('entity.meeting.group');
    }


}
