<?php

namespace App\Controller\Admin\Crud;

use App\Controller\Educator\Crud\StudentCrudEducatorController;
use App\Entity\Room;
use App\Entity\User;
use Doctrine\ORM\QueryBuilder;
use EasyCorp\Bundle\EasyAdminBundle\Collection\FieldCollection;
use EasyCorp\Bundle\EasyAdminBundle\Collection\FilterCollection;
use EasyCorp\Bundle\EasyAdminBundle\Config\Action;
use EasyCorp\Bundle\EasyAdminBundle\Config\Actions;
use EasyCorp\Bundle\EasyAdminBundle\Config\Crud;
use EasyCorp\Bundle\EasyAdminBundle\Config\Filters;
use EasyCorp\Bundle\EasyAdminBundle\Controller\AbstractCrudController;
use EasyCorp\Bundle\EasyAdminBundle\Dto\EntityDto;
use EasyCorp\Bundle\EasyAdminBundle\Dto\SearchDto;
use EasyCorp\Bundle\EasyAdminBundle\Field\AssociationField;
use EasyCorp\Bundle\EasyAdminBundle\Field\IdField;
use EasyCorp\Bundle\EasyAdminBundle\Field\NumberField;
use EasyCorp\Bundle\EasyAdminBundle\Field\TextField;
use Symfony\Component\Security\Http\Attribute\IsGranted;

#[IsGranted(User::ROLE_ADMIN)]
class RoomCrudController extends AbstractCrudController
{
    public static function getEntityFqcn(): string
    {
        return Room::class;
    }
    public function configureCrud(Crud $crud): Crud
    {
        return $crud
            ->setEntityLabelInSingular('entity.room.singular_name')
            ->setEntityLabelInPlural('entity.room.plural_name')
            ->setDefaultSort(['floor' => 'ASC', 'doorNumber' => 'ASC'])
            ->setPageTitle(Crud::PAGE_DETAIL, 'entity.room.singular_name')
            ;
    }
    public function configureFilters(Filters $filters): Filters
    {
        return $filters
            ->add('floor')
            ->add('doorNumber')
            ;
    }
    public function configureActions(Actions $actions): Actions
    {
        return $actions
            ->remove(Crud::PAGE_DETAIL, Action::DELETE)
            ->remove(Crud::PAGE_INDEX, Action::DELETE)
            ->disable(Action::BATCH_DELETE, Action::DELETE)
            ;
    }
    public function createIndexQueryBuilder(SearchDto $searchDto, EntityDto $entityDto, FieldCollection $fields, FilterCollection $filters): QueryBuilder
    {
        return parent::createIndexQueryBuilder($searchDto, $entityDto, $fields, $filters)
            ->addSelect('student_index')
            ->leftJoin('entity.students', 'student_index');
    }
    public function configureFields(string $pageName): iterable
    {
        $controller = in_array(User::ROLE_ADMIN, $this->getUser()->getRoles()) ? StudentCrudController::class : StudentCrudEducatorController::class;

        return [
            IdField::new('id')
                ->setLabel('entity.id')
                ->hideOnForm(),
            NumberField::new('floor')
                ->setLabel('entity.room.floor')
                ->hideOnForm()
                ->setColumns('col-12'),
            NumberField::new('doorNumber')
                ->setLabel('entity.room.door_number')
                ->setColumns('col-12'),
            AssociationField::new('students')
                ->setLabel('entity.student.plural_name')
                ->hideOnIndex()
                ->autocomplete()
                ->setCrudController($controller)
                ->setColumns('col-12')
                ->setTemplatePath('admin/show_students.html.twig')
                ->setQueryBuilder(function (QueryBuilder $qb) {
                    $qb
                        ->leftJoin('entity.user', 'user_query_autocomplete')
                        ->andWhere('user_query_autocomplete.active = :active')
                        ->andWhere('entity.room IS NULL')
                        ->setParameter('active', true);
                })
                ->setFormTypeOptions([
                    'by_reference' => false,
                ]),
            TextField::new('studentsCount')
                ->setLabel('entity.room.student_count')
                ->setFormTypeOption('mapped', false)
                ->hideOnDetail()
                ->hideOnForm()
                ->setValue('0')
                ->formatValue(function ($value, $entity) {
                    return $this->countAllStudentsByRoomId($entity);
                }),
        ];
    }
    protected function countAllStudentsByRoomId($entity): int
    {
        return $entity->getStudents()->count();
    }

}
