<?php

namespace App\Controller\Admin\Crud;

use App\Controller\Admin\Filter\ActiveUserFilter;
use App\Controller\Admin\Filter\RoomFloorFilter;
use App\Controller\Educator\Crud\StudentNoteCrudEducatorController;
use App\Entity\Guardian;
use App\Entity\Student;
use App\Entity\User;
use DateTimeImmutable;
use Doctrine\ORM\QueryBuilder;
use EasyCorp\Bundle\EasyAdminBundle\Collection\FieldCollection;
use EasyCorp\Bundle\EasyAdminBundle\Collection\FilterCollection;
use EasyCorp\Bundle\EasyAdminBundle\Config\Action;
use EasyCorp\Bundle\EasyAdminBundle\Config\Actions;
use EasyCorp\Bundle\EasyAdminBundle\Config\Crud;
use EasyCorp\Bundle\EasyAdminBundle\Config\Filters;
use EasyCorp\Bundle\EasyAdminBundle\Controller\AbstractCrudController;
use EasyCorp\Bundle\EasyAdminBundle\Dto\BatchActionDto;
use EasyCorp\Bundle\EasyAdminBundle\Dto\EntityDto;
use EasyCorp\Bundle\EasyAdminBundle\Dto\SearchDto;
use EasyCorp\Bundle\EasyAdminBundle\Field\AssociationField;
use EasyCorp\Bundle\EasyAdminBundle\Field\ChoiceField;
use EasyCorp\Bundle\EasyAdminBundle\Field\CountryField;
use EasyCorp\Bundle\EasyAdminBundle\Field\DateField;
use EasyCorp\Bundle\EasyAdminBundle\Field\EmailField;
use EasyCorp\Bundle\EasyAdminBundle\Field\FormField;
use EasyCorp\Bundle\EasyAdminBundle\Field\IdField;
use EasyCorp\Bundle\EasyAdminBundle\Field\TelephoneField;
use EasyCorp\Bundle\EasyAdminBundle\Field\TextField;
use EasyCorp\Bundle\EasyAdminBundle\Router\AdminUrlGenerator;
use Psr\Container\ContainerExceptionInterface;
use Psr\Container\NotFoundExceptionInterface;
use Symfony\Component\HttpFoundation\RedirectResponse;
use Symfony\Component\Security\Http\Attribute\IsGranted;

#[IsGranted(User::ROLE_ADMIN)]
class StudentCrudController extends AbstractCrudController
{
    private AdminUrlGenerator $adminUrlGenerator;

    public function __construct(AdminUrlGenerator $adminUrlGenerator)
    {
        $this->adminUrlGenerator = $adminUrlGenerator;
    }

    public static function getEntityFqcn(): string
    {
        return Student::class;
    }
    public function createEntity(string $entityFqcn): Student
    {
        $student = new Student();
        $user = new User();
        $user->setRoles([User::ROLE_STUDENT]);
        $student->setUser($user);

        return $student;
    }

    public function createIndexQueryBuilder(SearchDto $searchDto, EntityDto $entityDto, FieldCollection $fields, FilterCollection $filters): QueryBuilder
    {
        return parent::createIndexQueryBuilder($searchDto, $entityDto, $fields, $filters)
            ->addSelect('user_index')
            ->leftJoin('entity.user', 'user_index');
    }

    public function configureCrud(Crud $crud): Crud
    {
        return $crud
            ->setEntityLabelInSingular('entity.student.singular_name')
            ->setEntityLabelInPlural('entity.student.plural_name')
            ->setDefaultSort(['id' => 'ASC',])
            ->setSearchFields(['user.firstName', 'user.lastName', 'user.email', 'user.phoneNumber'])
            ;
    }
    public function configureActions(Actions $actions): Actions
    {
        $showAttendance = Action::new('attendance','entity.student.show_attendance')
            ->linkToRoute('app_attendance_view', function ($entity) {
                return ['id' => $entity->getId()];
            });
        $importStudents = Action::new('importStudents', 'entity.student.import_students')
            ->linkToRoute('app_import_students')
            ->createAsGlobalAction()
            ->setCssClass('btn btn-primary');
        $importTraining = Action::new('importTraining', 'entity.student.import_trainings')
            ->linkToRoute('app_import_trainings')
            ->createAsGlobalAction()
            ->setCssClass('btn btn-primary');
        $exportStudents = Action::new('exportStudents', 'entity.student.export_students')
            ->linkToRoute('app_export_students')
            ->createAsGlobalAction()
            ->setCssClass('btn btn-primary');
        $batchInactiveStudents = Action::new('batchInactiveStudents', 'entity.student.action_inactive_students')
            ->linkToCrudAction('inactiveStudents')
            ->setCssClass('btn btn-primary');

        $newNote = Action::new('newNote', 'entity.student.give_student_note')
            ->linkToRoute('app_student_note_create', function ($entity) {
            return ['id' => $entity->getId()];
        });
        $showNotes = Action::new('showNotes', 'entity.student.show_student_notes')->linkToUrl(function ($entity) {
            $controller = in_array(User::ROLE_ADMIN, $this->getUser()->getRoles()) ? StudentNoteCrudController::class : StudentNoteCrudEducatorController::class;
            /** @var User $user */
            $user = $this->getUser();
            return $this->adminUrlGenerator
                ->setController($controller)
                ->setDashboard($user->getDashboard())
                ->setAction(Action::INDEX)
                ->set('filters[student][comparison]', '=')
                ->set('filters[student][value]', $entity->getId())
                ->generateUrl();
        });

        return $actions
            ->add(Crud::PAGE_INDEX, $importStudents)
            ->add(Crud::PAGE_INDEX, $importTraining)
            ->add(Crud::PAGE_INDEX, $showAttendance)
            ->add(Crud::PAGE_DETAIL, $showAttendance)
            ->add(Crud::PAGE_INDEX, $newNote)
            ->add(Crud::PAGE_DETAIL, $newNote)
            ->add(Crud::PAGE_INDEX, $showNotes)
            ->add(Crud::PAGE_DETAIL, $showNotes)
            ->add(Crud::PAGE_INDEX, $exportStudents)
            ->addBatchAction($batchInactiveStudents)
            ->setPermission('newNote', User::ROLE_EDUCATOR)
            ->setPermission('exportStudents', User::ROLE_ADMIN)
            ->setPermission('batchInactiveStudents', User::ROLE_ADMIN)
            ->remove(Crud::PAGE_INDEX, Action::DELETE)
            ->remove(Crud::PAGE_DETAIL, Action::DELETE)
            ->disable(Action::BATCH_DELETE, Action::DELETE)
            ;
    }
    public function configureFilters(Filters $filters): Filters
    {
        return $filters
            ->add('educator')
            ->add(RoomFloorFilter::new('floor', 'entity.room.floor'))
            ->add(ActiveUserFilter::new('activeUser', 'entity.user.active.name'))
            ;
    }
    public function configureFields(string $pageName): iterable
    {
        return [
            FormField::addTab('entity.student.personal_info'),
            IdField::new('id')
                ->setLabel('entity.id')
                ->hideOnForm(),
            AssociationField::new('user')
                ->setColumns('col-12')
                ->renderAsEmbeddedForm()
                ->onlyOnForms()
                ->setFormTypeOptions([
                    'label' => false,
                ]),
            TextField::new('user.firstName')
                ->setLabel('entity.user.first_name')
                ->hideOnForm(),
            TextField::new('user.lastName')
                ->setLabel('entity.user.last_name')
                ->hideOnForm(),
            EmailField::new('user.email')
                ->setLabel('entity.user.email')
                ->hideOnForm(),
            DateField::new('dateOfBirth')
                ->hideOnIndex()
                ->setLabel('entity.student.date_of_birth')
                ->setFormTypeOption('attr', ['max' => (new DateTimeImmutable())->format('Y-m-d')])
                ->setColumns('col-12'),
            CountryField::new('user.phoneCountry')
                ->setLabel('entity.user.phone_country')
                ->setColumns('col-6')
                ->hideOnIndex()
                ->hideOnForm()
                ->setRequired(false),
            TelephoneField::new('user.phoneNumber')
                ->setColumns('col-6')
                ->hideOnForm()
                ->setLabel('entity.user.phone_number'),
            ChoiceField::new('user.active')
                ->setLabel('entity.user.active.name')
                ->setColumns('col-12')
                ->hideOnForm()
                ->setChoices([
                    'entity.user.active.no' => 0,
                    'entity.user.active.yes' => 1]),

            AssociationField::new('guardians')
                ->setLabel('entity.guardian.plural_name')
                ->setColumns('col-12')
                ->hideOnIndex()
                ->autocomplete()
                ->setTemplatePath('admin/show_guardians.html.twig'),
            FormField::addTab('entity.student.address'),
            TextField::new('street')
                ->setLabel('entity.student.street')
                ->setColumns('col-12')
                ->hideOnIndex(),
            TextField::new('postalCode')
                ->setLabel('entity.student.postal_code')
                ->setColumns('col-3')
                ->hideOnIndex(),
            TextField::new('city')
                ->setLabel('entity.student.city')
                ->setColumns('col-6')
                ->hideOnIndex(),
            CountryField::new('country')
                ->setColumns('col-3')
                ->setLabel('entity.student.country')
                ->setFormTypeOptions(
                    [
                        'choice_translation_domain' => false,
                    ]
                ),
            FormField::addTab('entity.student.additional_info'),
            AssociationField::new('dormitoryCards')
                ->setLabel('entity.dormitory_card.singular_name')
                ->setColumns('col-12')
                ->hideOnForm()
                ->hideOnIndex()
                ->setTemplatePath('admin/show_dormitory_card_number.html.twig'),
            AssociationField::new('educator')
                ->setLabel('entity.educator.singular_name')
                ->setCrudController(EducatorCrudController::class)
                ->autocomplete()
                ->setQueryBuilder(function (QueryBuilder $qb) {
                    $qb->leftJoin('entity.user', 'u')
                        ->andWhere('u.active = :isActive')
                        ->setParameter('isActive', true);
                })
                ->setColumns('col-12'),
            AssociationField::new('room')
                ->setLabel('entity.room.singular_name')
                ->hideOnIndex()
                ->setCrudController(RoomCrudController::class)
                ->autocomplete()
                ->setColumns('col-12'),
            AssociationField::new('classroom')
                ->setLabel('entity.classroom.singular_name')
                ->autocomplete()
                ->setCrudController(ClassroomCrudController::class)
                ->hideOnIndex()
                ->setColumns('col-12'),
            AssociationField::new('masterTeacher')
                ->setLabel('entity.master_teacher.singular_name')
                ->autocomplete()
                ->setCrudController(MasterTeacherCrudController::class)
                ->setColumns('col-12')
                ->hideOnIndex(),
            AssociationField::new('coaches')
                ->setLabel('entity.coach.plural_name')
                ->autocomplete()
                ->setColumns('col-12')
                ->hideOnIndex()
                ->setTemplatePath('admin/show_coaches.html.twig')
                ->setFormTypeOptions([
                    'by_reference' => false,
                ]),
            AssociationField::new('activities')
                ->setLabel('entity.activity.plural_name')
                ->autocomplete()
                ->setColumns('col-12')
                ->hideOnIndex()
                ->hideOnForm()
                ->setTemplatePath('admin/show_activities.html.twig')
                ->setFormTypeOptions([
                    'by_reference' => false,
                ]),
        ];
    }

    /**
     * @throws NotFoundExceptionInterface
     * @throws ContainerExceptionInterface
     */
    public function inactiveStudents(BatchActionDto $batchActionDto): RedirectResponse
    {
        $className = $batchActionDto->getEntityFqcn();
        $entityManager = $this->container->get('doctrine')->getManagerForClass($className);
        foreach ($batchActionDto->getEntityIds() as $id) {
            /** @var Student $student */
            $student = $entityManager->find($className, $id);
            $student->getUser()->setActive(false);
            foreach ($student->getDormitoryCards() as $dormitoryCard) {
                $dormitoryCard->setStudent(null);
            }
            foreach ($student->getGuardians() as $guardian) {
                $activeChildren = $this->getActiveChildren($guardian);
                if (count($activeChildren) === 0) {
                    $guardian->getUser()->setActive(false);
                }
            }
        }

        $entityManager->flush();

        return $this->redirect($batchActionDto->getReferrerUrl());
    }
    private function getActiveChildren(Guardian $guardian): array
    {
        $activeChildren = [];
        foreach ($guardian->getChildren() as $child) {
            if ($child->getUser()->isActive()) {
                $activeChildren[] = $child;
            }
        }
        return $activeChildren;
    }
}
