<?php

namespace App\Controller\Admin\Crud;

use App\Controller\Admin\Filter\DayInWeekFilter;
use App\Controller\Educator\Crud\CoachCrudEducatorController;
use App\Entity\Activity;
use App\Services\GeneralFunctionsService;
use Doctrine\ORM\QueryBuilder;
use EasyCorp\Bundle\EasyAdminBundle\Collection\FieldCollection;
use EasyCorp\Bundle\EasyAdminBundle\Collection\FilterCollection;
use EasyCorp\Bundle\EasyAdminBundle\Config\Action;
use EasyCorp\Bundle\EasyAdminBundle\Config\Actions;
use EasyCorp\Bundle\EasyAdminBundle\Config\Crud;
use EasyCorp\Bundle\EasyAdminBundle\Config\Filters;
use EasyCorp\Bundle\EasyAdminBundle\Controller\AbstractCrudController;
use EasyCorp\Bundle\EasyAdminBundle\Dto\EntityDto;
use EasyCorp\Bundle\EasyAdminBundle\Dto\SearchDto;
use EasyCorp\Bundle\EasyAdminBundle\Field\AssociationField;
use EasyCorp\Bundle\EasyAdminBundle\Field\ChoiceField;
use EasyCorp\Bundle\EasyAdminBundle\Field\IdField;
use EasyCorp\Bundle\EasyAdminBundle\Field\TimeField;
use Exception;
use Symfony\Component\ExpressionLanguage\Expression;
use Symfony\Component\Security\Http\Attribute\IsGranted;

#[IsGranted(new Expression('is_granted("ROLE_ADMIN") or is_granted("ROLE_EDUCATOR")'))]
class TrainingCrudController extends AbstractCrudController
{
    public static function getEntityFqcn(): string
    {
        return Activity::class;
    }
    public function createEntity(string $entityFqcn): Activity
    {
        $activity = new Activity();
        $activity->setType(Activity::TRAINING);
        return $activity;
    }
    public function createIndexQueryBuilder(SearchDto $searchDto, EntityDto $entityDto, FieldCollection $fields, FilterCollection $filters): QueryBuilder
    {
        return parent::createIndexQueryBuilder($searchDto, $entityDto, $fields, $filters)
            ->andWhere('entity.type = :type')
            ->setParameter('type', Activity::TRAINING)
            ->addSelect('user_index', 'coach_index')
            ->leftJoin('entity.coach', 'coach_index')
            ->leftJoin('coach_index.user', 'user_index');
    }
    public function configureCrud(Crud $crud): Crud
    {
        return $crud
            ->setEntityLabelInSingular('entity.activity.type.training.singular_name')
            ->setEntityLabelInPlural('entity.activity.type.training.plural_name')
            ->setSearchFields(['coach.user.firstName', 'coach.user.lastName', 'coach.user.email'])
            ->setPageTitle(Crud::PAGE_DETAIL, 'entity.activity.type.training.singular_name')
            ;
    }
    public function configureFilters(Filters $filters): Filters
    {
        return $filters
            ->add('students')
            ->add('coach')
            ->add(DayInWeekFilter::new('dayInWeek', 'general.day'))
            ;
    }
    public function configureActions(Actions $actions): Actions
    {
        $importTraining = Action::new('importTraining', 'entity.student.import_trainings')
            ->linkToRoute('app_import_trainings')
            ->createAsGlobalAction()
            ->setCssClass('btn btn-primary');
        return $actions
            ->add(Crud::PAGE_INDEX, $importTraining)
            ;
    }

    /**
     * @throws Exception
     */
    public function configureFields(string $pageName): iterable
    {
        $choices = [];
        for ($i = 0; $i < 7; $i++) {
            $choices[$i] = GeneralFunctionsService::dayInWeekToString($i);
        }
        $controller = in_array('ROLE_ADMIN', $this->getUser()->getRoles()) ? CoachCrudController::class : CoachCrudEducatorController::class;

        return [
            IdField::new('id')
                ->setLabel('entity.id')
                ->hideOnForm(),
            AssociationField::new('coach')
                ->setLabel('entity.coach.singular_name')
                ->setColumns('col-12')
                ->setCrudController($controller)
                ->autocomplete()
                ->setRequired(true),
            AssociationField::new('students')
                ->setLabel('entity.student.plural_name')
                ->hideOnIndex()
                ->autocomplete()
                ->setColumns('col-12')
                ->setRequired(true)
                ->setTemplatePath('admin/show_students.html.twig'),
            TimeField::new('fromTime')
                ->setLabel('entity.activity.from')
                ->setColumns('col-12')
                ->setRequired(true),
            TimeField::new('toTime')
                ->setLabel('entity.activity.to')
                ->setColumns('col-12')
                ->setRequired(true),
            ChoiceField::new('dayInWeek')
                ->setLabel('general.day')
                ->setColumns('col-12')
                ->setRequired(true)
                ->setTranslatableChoices($choices)
        ];
    }

}
