<?php

namespace App\Controller\Admin\Crud;

use App\Controller\Educator\Crud\StudentCrudEducatorController;
use App\Entity\Attendance;
use App\Entity\User;
use DateTimeImmutable;
use Doctrine\ORM\QueryBuilder;
use EasyCorp\Bundle\EasyAdminBundle\Collection\FieldCollection;
use EasyCorp\Bundle\EasyAdminBundle\Collection\FilterCollection;
use EasyCorp\Bundle\EasyAdminBundle\Config\Crud;
use EasyCorp\Bundle\EasyAdminBundle\Config\Filters;
use EasyCorp\Bundle\EasyAdminBundle\Controller\AbstractCrudController;
use EasyCorp\Bundle\EasyAdminBundle\Dto\EntityDto;
use EasyCorp\Bundle\EasyAdminBundle\Dto\SearchDto;
use EasyCorp\Bundle\EasyAdminBundle\Field\AssociationField;
use EasyCorp\Bundle\EasyAdminBundle\Field\DateTimeField;
use EasyCorp\Bundle\EasyAdminBundle\Field\IdField;
use EasyCorp\Bundle\EasyAdminBundle\Field\TextField;
use Symfony\Component\ExpressionLanguage\Expression;
use Symfony\Component\Security\Http\Attribute\IsGranted;

#[IsGranted(new Expression('is_granted("ROLE_ADMIN") or is_granted("ROLE_EDUCATOR")'))]
class WalkOutCrudController extends AbstractCrudController
{
    public static function getEntityFqcn(): string
    {
        return Attendance::class;
    }
    public function createEntity(string $entityFqcn): Attendance
    {
        $attendance = new Attendance();
        $attendance->setAction(Attendance::WALK);

        return $attendance;
    }
    public function configureCrud(Crud $crud): Crud
    {
        return $crud
            ->setEntityLabelInSingular('entity.attendance.action.walk_out.singular_name')
            ->setEntityLabelInPlural('entity.attendance.action.walk_out.plural_name')
            ->setDefaultSort(['time' => 'DESC',])
            ->setSearchFields(['student.user.firstName', 'student.user.lastName', 'student.user.email'])
            ->setPageTitle(Crud::PAGE_DETAIL, 'entity.attendance.action.walk_out.singular_name')
            ;
    }
    public function createIndexQueryBuilder(SearchDto $searchDto, EntityDto $entityDto, FieldCollection $fields, FilterCollection $filters): QueryBuilder
    {
        return parent::createIndexQueryBuilder($searchDto, $entityDto, $fields, $filters)
            ->andWhere('entity.action = :action')
            ->setParameter('action', Attendance::WALK)
            ->addSelect('user_index', 'student_index')
            ->leftJoin('entity.student', 'student_index')
            ->leftJoin('student_index.user', 'user_index');
    }
    public function configureFilters(Filters $filters): Filters
    {
        return $filters
            ->add('student')
            ->add('time')
            ;
    }

    public function configureFields(string $pageName): iterable
    {
        $date = new DateTimeImmutable();
        $controller = in_array(User::ROLE_ADMIN, $this->getUser()->getRoles()) ? StudentCrudController::class : StudentCrudEducatorController::class;
        return [
            IdField::new('id')
                ->setLabel('entity.id')
                ->hideOnForm(),
            DateTimeField::new('time')
                ->setLabel('entity.attendance.time')
                ->setRequired(true)
                ->setFormTypeOption('attr', [
                    'min' => $date->format('Y-m-d H:i'),
                ])
                ->setColumns('col-12'),
            DateTimeField::new('validUntil')
                ->setLabel('entity.attendance.valid_until')
                ->setFormTypeOption('attr', [
                    'min' => $date->format('Y-m-d H:i'),
                ])
                ->setRequired(true)
                ->setColumns('col-12'),
            AssociationField::new('student')
                ->setLabel('entity.student.singular_name')
                ->setCrudController($controller)
                ->setRequired(true)
                ->autocomplete()
                ->setColumns('col-12'),
            TextField::new('note')
                ->setLabel('entity.attendance.note')
                ->hideOnIndex()
                ->setColumns('col-12'),
        ];
    }
}
