<?php

namespace App\Controller;

use App\Entity\Attendance;
use App\Entity\Student;
use App\Entity\User;
use App\Form\Attendance\ShowAttendanceType;
use App\Form\Attendance\ShowStudentAttendanceGuardianType;
use App\Form\Attendance\ShowStudentAttendanceType;
use App\Repository\AttendanceRepository;
use App\Repository\StudentRepository;
use App\Services\DailyAttendance;
use DateInterval;
use DatePeriod;
use DateTimeImmutable;
use Doctrine\ORM\EntityManagerInterface;
use Doctrine\ORM\EntityNotFoundException;
use Exception;
use Symfony\Bundle\FrameworkBundle\Controller\AbstractController;
use Symfony\Component\ExpressionLanguage\Expression;
use Symfony\Component\HttpFoundation\JsonResponse;
use Symfony\Component\HttpFoundation\Request;
use Symfony\Component\HttpFoundation\Response;
use Symfony\Component\HttpKernel\Exception\AccessDeniedHttpException;
use Symfony\Component\Routing\Annotation\Route;
use Symfony\Component\Security\Http\Attribute\IsGranted;

#[Route('/dochadzka', name: 'app_attendance_')]
#[IsGranted('ROLE_USER')]
class AttendanceController extends AbstractController
{
    private EntityManagerInterface $entityManager;

    public function __construct(EntityManagerInterface $entityManager)
    {
        $this->entityManager = $entityManager;
    }
    /**
     * @throws Exception
     */
    #[Route('/', name: 'index')]
    #[IsGranted(new Expression('is_granted("ROLE_ADMIN") or is_granted("ROLE_EDUCATOR")'))]
    public function index(int $app_starting_year, AttendanceRepository $attendanceRepository,
                          StudentRepository $studentRepository, Request $request): Response
    {
        /** @var User $user */
        $day = new DateTimeImmutable();
        $educator = $this->getUser()->getEducator();
        $form = $this->createForm(ShowAttendanceType::class,[], [
            'data' => [
                'educator' => $educator,
                'date' => $day,
                'start' => $app_starting_year,
            ],
        ]);

        $form->handleRequest($request);
        if ($form->isSubmitted() && $form->isValid()) {
            $floor = $form->get('floor')->getData();
            $educator = $form->get('educator')->getData();
            $day = $form->get('date')->getData();
            $wholeMonth = $form->get('wholeMonth')->getData();
            $students = $studentRepository->getStudentsByFloorOrEducatorOrBoth($floor, $educator);
            if (empty($students)) {
                $this->addFlash('warning', 'entity.attendance.form_empty_for_educator_or_floor');
            }
            if ($wholeMonth) {
                $days = $this->daysInMonthOrToNow($day);
            }
            else {
                $days[] = $day;
            }

            foreach ($days as $day) {
                $this->setAttendancesToStudents($day, $students,
                    $attendanceRepository->findByStudentsArrayAndDate($students, $day),
                    $attendanceRepository->findLastPresentOrAbsentFromArray($students, $day),
                    $attendanceRepository->findEveningCheck($students, $day)
                );
            }
            /** @var Student $student */
            foreach ($students as $student) {
                foreach ($student->getActivities() as $activity) {
                    /** @var DailyAttendance $dailyAttendance */
                    foreach ($student->getDailyAttendances() as $dailyAttendance) {

                        if ((int)$dailyAttendance->getDay()->format('w') === $activity->getDayInWeek()) {
                            $dailyAttendance->addActivity($activity);
                        }
                    }
                }
            }

            return $this->render('attendance/index.html.twig', [
                'form' => $form,
                'students' => $students,
                'days' => $days,
            ]);
        }

        return $this->render('attendance/index.html.twig', [
            'form' => $form,
            'students' => null,
        ]);
    }
    #[Route('/zmen-data', name: 'update', methods: ['POST'])]
    public function update(StudentRepository $studentRepository, Request $request): JsonResponse
    {
        $parameters = json_decode($request->getContent());
        $currentDate = new DateTimeImmutable();

        $student = $studentRepository->findOneBy(['id' => $parameters->studentId]);
        if ($student === null) {
            throw $this->createNotFoundException();
        }
        $attendance = new Attendance();
        $attendance->setStudent($student);
        $attendance->setTime($currentDate);

        if ($parameters->action === Attendance::ABSENT_ACTION) {
            $attendance->setAction(Attendance::ABSENT);
        } elseif ($parameters->action === Attendance::PRESENT_ACTION) {
            $attendance->setAction(Attendance::PRESENT);
        }
        $this->entityManager->persist($attendance);

        if ($currentDate->format('H') >= 20) {
            $checked = new Attendance();
            $checked->setStudent($student);
            $checked->setTime($currentDate);
            $checked->setAction(Attendance::EVENING_CHECK);
            $this->entityManager->persist($checked);
        }
        $this->entityManager->flush();

        return new JsonResponse([
            'attendanceTime' => $currentDate,
            'checked' => isset($checked),
        ]);
    }


    /**
     * @throws Exception
     */
    #[Route('/prehlad', name: 'view')]
    public function view(int $app_starting_year, StudentRepository $studentRepository, AttendanceRepository $attendanceRepository, Request $request): Response
    {
        /** @var User $user */
        $user = $this->getUser();
        if ($user->getStudent() !== null && $user->getStudent()->getId() !== (int)$request->get('id')){
            throw new AccessDeniedHttpException();
        }
        $student = $studentRepository->findOneBy(['id' => $request->get('id')]);
        if ($student === null) {
            throw new EntityNotFoundException();
        }

        $date = new DateTimeImmutable();
        $form = $this->createForm(ShowStudentAttendanceType::class,[], [
            'data' => [
                'date' => $date,
                'start' => $app_starting_year,
            ],
        ]);
        $form->handleRequest($request);
        if ($form->isSubmitted() && $form->isValid()) {
            $year = $form->get('year')->getData();
            $month = $form->get('month')->getData();
            $date = new DateTimeImmutable("$year-$month");
        }
        $days = $this->daysInMonthOrToNow($date);
        $this->setAttendanceForEachDay($days, $student, $attendanceRepository);
        return $this->render('attendance/show_student.html.twig', [
            'student' => $student,
            'form' => $form,
            'days' => $days,
        ]);
    }

    /**
     * @throws Exception
     */
    #[Route('/prehlad-deti', name: 'view_children')]
    #[IsGranted(User::ROLE_GUARDIAN)]
    public function viewChildren(int $app_starting_year, AttendanceRepository $attendanceRepository, Request $request): Response
    {
        $date = new DateTimeImmutable();
        $form = $this->createForm(ShowStudentAttendanceGuardianType::class,[], [
            'data' => [
                'date' => $date,
                'start' => $app_starting_year,
            ],
        ]);

        $form->handleRequest($request);
        if ($form->isSubmitted() && $form->isValid()) {
            $year = $form->get('year')->getData();
            $month = $form->get('month')->getData();
            $date = new DateTimeImmutable("$year-$month");
            $student = $form->get('child')->getData();
            $days = $this->daysInMonthOrToNow($date);
            $this->setAttendanceForEachDay($days, $student, $attendanceRepository);
            return $this->render('attendance/show_student.html.twig', [
                'student' => $student,
                'form' => $form,
                'days' => $days,
            ]);
        }
        return $this->render('basic_form.html.twig', [
            'form' => $form,
            'templateUrl' => null,
        ]);
    }

    /**
     * @throws Exception
     */
    public function daysInMonthOrToNow(DateTimeImmutable $dateTimeImmutable): DatePeriod
    {
        $firstDayOfMonth = new DateTimeImmutable('first day of ' . $dateTimeImmutable->format('Y-m'));
        $lastDayOfMonth = new DateTimeImmutable('last day of ' . $dateTimeImmutable->format('Y-m'));
        $lastDayOfMonth = $lastDayOfMonth->modify('+1 day');
        if ($lastDayOfMonth > new DateTimeImmutable()) {
            $lastDayOfMonth = new DateTimeImmutable();
        }

        return new DatePeriod($firstDayOfMonth, new DateInterval('P1D'), $lastDayOfMonth);
    }
    public function setAttendanceForEachDay(DatePeriod $days, Student $student, AttendanceRepository $attendanceRepository): void
    {
        foreach ($days as $day) {
            $this->setAttendancesToStudents($day, [$student],
                $attendanceRepository->findByStudentAndDate($student, $day),
                $attendanceRepository->findLastPresentOrAbsentFromArray([$student], $day),
                $attendanceRepository->findEveningCheck([$student], $day)
            );
        }
    }
    private function setAttendancesToStudents(DateTimeImmutable $day, array $students, array $attendancesByStudentsAndDate, array $currentAttendances, array $eveningChecks): void
    {
        /** @var Student $student */
        foreach ($students as $student) {
            $dailyAttendance = $student->getDailyAttendance($day);
            /** @var Attendance $attendance */
            foreach ($attendancesByStudentsAndDate as $attendance) {
                if ($attendance->getStudent()->getId() === $student->getId()) {
                    $dailyAttendance->addOtherAttendance($attendance);
                }
            }

            //only one $attendance per student => break after it is found
            /** @var Attendance $attendance */
            foreach ($currentAttendances as $attendance) {
                if ($attendance === null) {
                    continue;
                }
                if ($attendance->getStudent()->getId() === $student->getId()) {
                    $dailyAttendance->setCurrentAttendance($attendance);
                    break;
                }
            }

            //only one $attendance per student => break after it is found
            /** @var Attendance $attendance */
            foreach ($eveningChecks as $attendance) {
                if ($attendance->getStudent()->getId() === $student->getId()) {
                    $dailyAttendance->setEveningCheck($attendance->getTime());
                    break;
                }
            }
        }
    }
}
