<?php

namespace App\Controller\Educator\Crud;


use App\Controller\Admin\Crud\StudentCrudController;
use App\Entity\Student;
use App\Entity\User;
use DateTimeImmutable;
use Doctrine\ORM\QueryBuilder;
use EasyCorp\Bundle\EasyAdminBundle\Config\Action;
use EasyCorp\Bundle\EasyAdminBundle\Config\Actions;
use EasyCorp\Bundle\EasyAdminBundle\Config\Crud;
use EasyCorp\Bundle\EasyAdminBundle\Field\AssociationField;
use EasyCorp\Bundle\EasyAdminBundle\Field\ChoiceField;
use EasyCorp\Bundle\EasyAdminBundle\Field\CountryField;
use EasyCorp\Bundle\EasyAdminBundle\Field\DateField;
use EasyCorp\Bundle\EasyAdminBundle\Field\EmailField;
use EasyCorp\Bundle\EasyAdminBundle\Field\FormField;
use EasyCorp\Bundle\EasyAdminBundle\Field\IdField;
use EasyCorp\Bundle\EasyAdminBundle\Field\IntegerField;
use EasyCorp\Bundle\EasyAdminBundle\Field\TelephoneField;
use EasyCorp\Bundle\EasyAdminBundle\Field\TextField;
use Symfony\Component\Security\Http\Attribute\IsGranted;

#[IsGranted(User::ROLE_EDUCATOR)]
class StudentCrudEducatorController extends StudentCrudController
{
    public static function getEntityFqcn(): string
    {
        return Student::class;
    }
    public function configureActions(Actions $actions): Actions
    {
        return parent::configureActions($actions)
            ->remove(Crud::PAGE_INDEX, Action::NEW)
            ->disable(Action::NEW)
            ;
    }
    public function configureFields(string $pageName): iterable
    {
        return [
            FormField::addTab('entity.student.personal_info'),
            IdField::new('id')
                ->setLabel('entity.id')
                ->hideOnForm(),
            AssociationField::new('user')
                ->setColumns('col-12')
                ->renderAsEmbeddedForm()
                ->onlyOnForms()
                ->setFormTypeOptions([
                    'label' => false,
                ]),
            TextField::new('user.firstName')
                ->setLabel('entity.user.first_name')
                ->hideOnForm(),
            TextField::new('user.lastName')
                ->setLabel('entity.user.last_name')
                ->hideOnForm(),
            EmailField::new('user.email')
                ->setLabel('entity.user.email')
                ->hideOnForm(),
            DateField::new('dateOfBirth')
                ->hideOnIndex()
                ->setLabel('entity.student.date_of_birth')
                ->setFormTypeOption('attr', ['max' => (new DateTimeImmutable())->format('Y-m-d')])
                ->setColumns('col-12'),
            CountryField::new('user.phoneCountry')
                ->setLabel('entity.user.phone_country')
                ->setColumns('col-6')
                ->hideOnIndex()
                ->hideOnForm()
                ->setRequired(false),
            TelephoneField::new('user.phoneNumber')
                ->setColumns('col-6')
                ->hideOnForm()
                ->setLabel('entity.user.phone_number'),
            ChoiceField::new('user.active')
                ->setLabel('entity.user.active.name')
                ->setColumns('col-12')
                ->hideOnForm()
                ->setChoices([
                    'entity.user.active.no' => 0,
                    'entity.user.active.yes' => 1]),

            AssociationField::new('guardians')
                ->setLabel('entity.guardian.plural_name')
                ->setColumns('col-12')
                ->hideOnIndex()
                ->autocomplete()
                ->setTemplatePath('admin/show_guardians.html.twig'),
            FormField::addTab('entity.student.address'),
            TextField::new('street')
                ->setLabel('entity.student.street')
                ->setColumns('col-12')
                ->hideOnIndex(),
            TextField::new('postalCode')
                ->setLabel('entity.student.postal_code')
                ->setColumns('col-3')
                ->hideOnIndex(),
            TextField::new('city')
                ->setLabel('entity.student.city')
                ->setColumns('col-6')
                ->hideOnIndex(),
            CountryField::new('country')
                ->setColumns('col-3')
                ->setLabel('entity.student.country')
                ->setFormTypeOptions(
                    [
                        'choice_translation_domain' => false,
                    ]
                ),
            FormField::addTab('entity.student.additional_info'),
            AssociationField::new('educator')
                ->setLabel('entity.educator.singular_name')
                ->setCrudController(EducatorCrudEducatorController::class)
                ->autocomplete()
                ->setQueryBuilder(function (QueryBuilder $qb) {
                    $qb->leftJoin('entity.user', 'u')
                        ->andWhere('u.active = :isActive')
                        ->setParameter('isActive', true);
                })
                ->setColumns('col-12'),
            AssociationField::new('room')
                ->setLabel('entity.room.singular_name')
                ->hideOnIndex()
                ->setCrudController(RoomCrudEducatorController::class)
                ->autocomplete()
                ->setColumns('col-12'),
            AssociationField::new('classroom')
                ->setLabel('entity.classroom.singular_name')
                ->autocomplete()
                ->setCrudController(ClassroomCrudEducatorController::class)
                ->hideOnIndex()
                ->setColumns('col-12'),
            AssociationField::new('masterTeacher')
                ->setLabel('entity.master_teacher.singular_name')
                ->autocomplete()
                ->setCrudController(MasterTeacherCrudEducatorController::class)
                ->setColumns('col-12')
                ->hideOnIndex(),
            AssociationField::new('coaches')
                ->setLabel('entity.coach.plural_name')
                ->autocomplete()
                ->setColumns('col-12')
                ->hideOnIndex()
                ->setTemplatePath('admin/show_coaches.html.twig')
                ->setFormTypeOptions([
                    'by_reference' => false,
                ]),
            AssociationField::new('activities')
                ->setLabel('entity.activity.plural_name')
                ->autocomplete()
                ->setColumns('col-12')
                ->hideOnIndex()
                ->hideOnForm()
                ->setTemplatePath('admin/show_activities.html.twig')
                ->setFormTypeOptions([
                    'by_reference' => false,
                ]),

        ];
    }
}
