<?php

namespace App\Controller\Educator;

use App\Controller\Admin\Crud\LeavingHomeCrudController;
use App\Controller\Admin\Crud\WalkOutCrudController;
use App\Controller\Educator\Crud\ClassroomCrudEducatorController;
use App\Controller\Educator\Crud\CoachCrudEducatorController;
use App\Controller\Educator\Crud\EducatorCrudEducatorController;
use App\Controller\Educator\Crud\GuardianCrudEducatorController;
use App\Controller\Educator\Crud\MasterTeacherCrudEducatorController;
use App\Controller\Educator\Crud\MeetingCrudEducatorController;
use App\Controller\Educator\Crud\RoomCrudEducatorController;
use App\Controller\Educator\Crud\SchoolCrudEducatorController;
use App\Controller\Educator\Crud\StudentCrudEducatorController;
use App\Controller\Educator\Crud\StudentNoteCrudEducatorController;
use App\Entity\Activity;
use App\Entity\Attendance;
use App\Entity\Classroom;
use App\Entity\Coach;
use App\Entity\Educator;
use App\Entity\Guardian;
use App\Entity\MasterTeacher;
use App\Entity\Meeting;
use App\Entity\Room;
use App\Entity\School;
use App\Entity\Student;
use App\Entity\StudentNote;
use App\Entity\User;
use App\Repository\MessageReceiverRepository;
use App\Repository\ScheduleRepository;
use App\Services\ScheduleService;
use DateTimeImmutable;
use EasyCorp\Bundle\EasyAdminBundle\Config\Action;
use EasyCorp\Bundle\EasyAdminBundle\Config\Actions;
use EasyCorp\Bundle\EasyAdminBundle\Config\Assets;
use EasyCorp\Bundle\EasyAdminBundle\Config\Crud;
use EasyCorp\Bundle\EasyAdminBundle\Config\Dashboard;
use EasyCorp\Bundle\EasyAdminBundle\Config\MenuItem;
use EasyCorp\Bundle\EasyAdminBundle\Config\UserMenu;
use EasyCorp\Bundle\EasyAdminBundle\Controller\AbstractDashboardController;
use EasyCorp\Bundle\EasyAdminBundle\Router\AdminUrlGenerator;
use Exception;
use Symfony\Component\HttpFoundation\Response;
use Symfony\Component\Routing\Annotation\Route;
use Symfony\Component\Security\Core\User\UserInterface;
use Symfony\Component\Security\Http\Attribute\IsGranted;
use Symfony\Contracts\Translation\TranslatorInterface;

#[IsGranted('ROLE_EDUCATOR')]
class EducatorDashboardController extends AbstractDashboardController
{
    private ScheduleRepository $scheduleRepository;
    private AdminUrlGenerator $adminUrlGenerator;
    private MessageReceiverRepository $messageReceiverRepository;
    private TranslatorInterface $translator;

    public function __construct(ScheduleRepository $scheduleRepository, AdminUrlGenerator $adminUrlGenerator, MessageReceiverRepository $messageReceiverRepository,
                                TranslatorInterface $translator)
    {
        $this->scheduleRepository = $scheduleRepository;
        $this->adminUrlGenerator = $adminUrlGenerator;
        $this->messageReceiverRepository = $messageReceiverRepository;
        $this->translator = $translator;
    }

    /**
     * @throws Exception
     */
    #[Route('/vychovavatel', name: 'educator')]
    public function index(): Response
    {
        $date = new DateTimeImmutable();
        $educatorUrl = $this->generateCrudUrl(EducatorCrudEducatorController::class);
        $studentUrl = $this->generateCrudUrl(StudentCrudEducatorController::class);
        $meetingUrl = $this->generateCrudUrl(MeetingCrudEducatorController::class);
        $walkUrl = $this->generateCrudUrl(WalkOutCrudController::class);
        $leaveUrl = $this->generateCrudUrl(LeavingHomeCrudController::class);
        $noteUrl = $this->generateCrudUrl(StudentNoteCrudEducatorController::class);
        $roomUrl = $this->generateCrudUrl(RoomCrudEducatorController::class);
        $messageUrl =  $this->adminUrlGenerator
            ->setDashboard(EducatorDashboardController::class)
            ->setRoute('app_message_show_received')
            ->generateUrl();
        $attendanceUrl = $this->adminUrlGenerator
            ->setDashboard(EducatorDashboardController::class)
            ->setRoute('app_attendance_index')
            ->generateUrl();
        return $this->render('dashboard.html.twig', [
            'schedules' => $this->scheduleRepository->findScheduleForDate($date),
            'currentDate' => $date,
            'freeDay' => (new ScheduleService)->checkIfFreeDay($date),
            'unreadMessagesCount' => $this->messageReceiverRepository->count(['isRead' => false, 'receiver' => $this->getUser()]),
            'url' => [
                'educator' => $educatorUrl,
                'student' => $studentUrl,
                'meeting' => $meetingUrl,
                'walk' => $walkUrl,
                'leave' => $leaveUrl,
                'note' => $noteUrl,
                'message' => $messageUrl,
                'room' => $roomUrl,
                'attendance' => $attendanceUrl,
            ]
        ]);
    }
    private function generateCrudUrl(string $class): string
    {
        return $this->adminUrlGenerator
            ->setDashboard(EducatorDashboardController::class)
            ->setController($class)
            ->setAction(Action::INDEX)
            ->generateUrl();
    }

    public function configureDashboard(): Dashboard
    {
        return Dashboard::new()
            ->setTitle($this->translator->trans('dashboard.main_name'));
    }
    public function configureCrud(): Crud
    {
        return Crud::new()
            // the max number of entities to display per page
            ->setPaginatorPageSize(30)
            // the number of pages to display on each side of the current page
            // e.g. if num pages = 35, current page = 7, and you set ->setPaginatorRangeSize(4)
            // the paginator displays: [Previous]  1 ... 3  4  5  6  [7]  8  9  10  11 ... 35  [Next]
            // set this number to 0 to display a simple "< Previous | Next >" pager
            ->setPaginatorRangeSize(4)
            ;
    }
    public function configureAssets(): Assets
    {
        return parent::configureAssets()
            ->addWebpackEncoreEntry('easyadmin');
    }
    public function configureActions(): Actions
    {
        return parent::configureActions()
            ->add(Crud::PAGE_INDEX, Action::DETAIL)
            ->update(Crud::PAGE_DETAIL, Action::EDIT, static function (Action $action) {
                return $action->setIcon('fa fa-edit');
            })
            ->update(Crud::PAGE_DETAIL, Action::INDEX, static function (Action $action) {
                return $action->setIcon('fa fa-list');
            })
            ;
    }
    public function configureMenuItems(): iterable
    {
        yield MenuItem::linkToDashboard('dashboard.main_panel', 'fa fa-home');
        yield MenuItem::linkToRoute('entity.message.plural_name', 'fas fa-envelope', 'app_message_show_received');
        yield MenuItem::linkToRoute('entity.schedule.singular_name', 'fas fa-calendar-days', 'app_schedule_index');

        yield MenuItem::linkToCrud('entity.educator.plural_name', 'fas fa-user-tie', Educator::class)
            ->setQueryParameter('filters[activeUser]', 1)
            ->setController(EducatorCrudEducatorController::class);

        yield MenuItem::section('');
        yield MenuItem::linkToCrud('entity.meeting.plural_name', 'fas fa-handshake-simple', Meeting::class)
            ->setQueryParameter('filters[educator][comparison]', '=')
            ->setQueryParameter('filters[educator][value]', $this->getUser()->getEducator()->getId())
            ->setController(MeetingCrudEducatorController::class);
        yield MenuItem::linkToCrud('entity.attendance.action.walk_out.plural_name', 'fas fa-shoe-prints', Attendance::class)
            ->setController(WalkOutCrudController::class);
        yield MenuItem::linkToCrud('entity.attendance.action.departure.plural_name', 'fas fa-paper-plane', Attendance::class)
            ->setController(LeavingHomeCrudController::class);
        yield MenuItem::linkToRoute('entity.attendance.singular_name', 'fas fa-clock', 'app_attendance_index');

        yield MenuItem::section('');

        yield MenuItem::subMenu('dashboard.accommodated_students', 'fas fa-users')
            ->setSubItems([
                MenuItem::linkToCrud('entity.student.plural_name', 'fas fa-user-graduate', Student::class)
                    ->setQueryParameter('filters[activeUser]', 1)
                    ->setQueryParameter('filters[educator][comparison]', '=')
                    ->setQueryParameter('filters[educator][value]', $this->getUser()->getEducator()->getId())
                    ->setController(StudentCrudEducatorController::class),
                MenuItem::linkToCrud('entity.guardian.plural_name', 'fas fa-hands-holding-child', Guardian::class)
                    ->setQueryParameter('filters[activeUser]', 1)
                    ->setController(GuardianCrudEducatorController::class),
                MenuItem::linkToCrud('entity.student_note.plural_name', 'fas fa-edit', StudentNote::class)
                    ->setController(StudentNoteCrudEducatorController::class),

            ]);
        yield MenuItem::linkToCrud('entity.room.plural_name', 'fas fa-bed', Room::class)
            ->setController(RoomCrudEducatorController::class);
        yield MenuItem::linkToCrud('entity.coach.plural_name', 'fas fa-person-running', Coach::class)
            ->setQueryParameter('filters[activeUser]', 1)
            ->setController(CoachCrudEducatorController::class);
        yield MenuItem::linkToCrud('entity.activity.type.training.plural_name', 'fas fa-basketball', Activity::class);
        yield MenuItem::linkToCrud('entity.master_teacher.plural_name', 'fas fa-chalkboard-user', MasterTeacher::class)
            ->setQueryParameter('filters[activeUser]', 1)
            ->setController(MasterTeacherCrudEducatorController::class);
        yield MenuItem::linkToCrud('entity.school.plural_name', 'fas fa-school', School::class)
            ->setController(SchoolCrudEducatorController::class);
                MenuItem::linkToCrud('entity.classroom.plural_name', 'fas fa-chalkboard', Classroom::class)
                    ->setController(ClassroomCrudEducatorController::class);
    }
    public function configureUserMenu(UserInterface $user): UserMenu
    {
        /** @var User $user */
        return parent::configureUserMenu($user)
            ->setName($user->getFirstName() . ' ' . $user->getLastName() . ' ' . $user->getEmail())
            ->addMenuItems([
                MenuItem::linkToRoute('entity.user.profile', 'fa fa-id-card', 'app_user_profile'),
            ]);
    }
}
