<?php

namespace App\Controller;

use App\Controller\Admin\AdminDashboardController;
use App\Controller\Admin\Crud\MeetingCrudController;
use App\Controller\Admin\Crud\StudentCrudController;
use App\Controller\Educator\Crud\MeetingCrudEducatorController;
use App\Controller\Educator\Crud\StudentCrudEducatorController;
use App\Entity\User;
use App\Form\Import\ImportGroupMeetingType;
use App\Form\Import\ImportIndividualMeetingType;
use App\Form\Import\ImportScheduleType;
use App\Form\Import\ImportStudentType;
use App\Form\Import\ImportTrainingType;
use App\Services\Import\ImportGroupMeetings;
use App\Services\Import\ImportIndividualMeetings;
use App\Services\Import\ImportSchedule;
use App\Services\Import\ImportStudents;
use App\Services\Import\ImportTraining;
use EasyCorp\Bundle\EasyAdminBundle\Config\Crud;
use EasyCorp\Bundle\EasyAdminBundle\Router\AdminUrlGenerator;
use Exception;
use Symfony\Bundle\FrameworkBundle\Controller\AbstractController;
use Symfony\Component\ExpressionLanguage\Expression;
use Symfony\Component\Form\Exception\TransformationFailedException;
use Symfony\Component\HttpFoundation\Request;
use Symfony\Component\HttpFoundation\Response;
use Symfony\Component\Routing\Annotation\Route;
use Symfony\Component\Security\Http\Attribute\IsGranted;
use Symfony\Component\Translation\TranslatableMessage;
use Symfony\Contracts\Translation\TranslatorInterface;

#[Route(['/import'], name: 'app_import_')]
#[IsGranted(new Expression('is_granted("ROLE_ADMIN") or is_granted("ROLE_EDUCATOR")'))]
class ImportController extends AbstractController
{
    private AdminUrlGenerator $adminUrlGenerator;

    public function __construct(AdminUrlGenerator $adminUrlGenerator)
    {
        $this->adminUrlGenerator = $adminUrlGenerator;
    }

    #[Route('/studenti', name: 'students')]
    public function importStudents (Request $request, ImportStudents $importStudents, TranslatorInterface $translator): Response
    {
        $form = $this->createForm(ImportStudentType::class);
        $form->handleRequest($request);
        if ($form->isSubmitted() && $form->isValid()) {
            $importedStudents = $form->get('uploadedFile')->getData();
            try {
                $importStudents->saveStudents($importedStudents);
                $this->addFlash(
                    'success', new TranslatableMessage('import.flash_student_success', [
                        '%count%' => count($importedStudents),
                ]));
                $controller = in_array(User::ROLE_ADMIN, $this->getUser()->getRoles()) ? StudentCrudController::class : StudentCrudEducatorController::class;
                /** @var User $user */
                $user = $this->getUser();
                $targetUrl = $this->adminUrlGenerator
                    ->setController($controller)
                    ->setDashboard($user->getDashboard())
                    ->setAction(Crud::PAGE_INDEX)
                    ->generateUrl();
                return $this->redirect($targetUrl);
            }
            catch (TransformationFailedException $exception) {
                $this->addFlash(
                    'danger',
                    new TranslatableMessage($exception->getInvalidMessage(), $exception->getInvalidMessageParameters(), 'validators'),
                );
            }
            catch (Exception $exception) {
                $this->addFlash(
                    'danger',
                    $exception->getMessage()
                );
            }
        }
        return $this->render('basic_form.html.twig', [
            'form' => $form,
            'templateUrl' => $this->generateUrl('app_download_template_students')
        ]);
    }
    #[Route('/individualne-stretnutia', name: 'individual_meeting')]
    public function individualMeeting(Request $request,ImportIndividualMeetings $importIndividualMeeting): Response
    {
        $form = $this->createForm(ImportIndividualMeetingType::class);
        $form->handleRequest($request);
        if ($form->isSubmitted() && $form->isValid()) {
            $importedIndividualMeetings = $form->get('uploadedFile')->getData();
            try {
                $importIndividualMeeting->saveMeeting($importedIndividualMeetings);
                $this->addFlash(
                    'success', new TranslatableMessage('import.flash_individual_meeting_success', [
                    '%count%' => count($importedIndividualMeetings),
                ]));
                $controller = in_array('ROLE_ADMIN', $this->getUser()->getRoles()) ? MeetingCrudController::class : MeetingCrudEducatorController::class;
                /** @var User $user */
                $user = $this->getUser();
                $targetUrl = $this->adminUrlGenerator
                    ->setDashboard($user->getDashboard())
                    ->setController($controller)
                    ->setAction(Crud::PAGE_INDEX)
                    ->generateUrl();
                return $this->redirect($targetUrl);
            }
            catch (TransformationFailedException $exception) {
                $this->addFlash(
                    'danger',
                    new TranslatableMessage($exception->getInvalidMessage(), $exception->getInvalidMessageParameters(), 'validators'),
                );
            }
            catch (Exception $exception) {
                $this->addFlash(
                    'danger',
                    $exception->getMessage()
                );
            }
        }
        return $this->render('basic_form.html.twig', [
            'form' => $form,
            'templateUrl' => $this->generateUrl('app_download_template_individual_meeting')
        ]);
    }

    #[Route('/skupinove-stretnutia', name: 'group_meeting')]
    public function groupMeeting(Request $request, ImportGroupMeetings $importGroupMeeting): Response
    {
        $form = $this->createForm(ImportGroupMeetingType::class);
        $form->handleRequest($request);
        if ($form->isSubmitted() && $form->isValid()) {
            $importedGroupMeetings = $form->get('uploadedFile')->getData();
            try {
                $importGroupMeeting->saveMeeting($importedGroupMeetings);
                $this->addFlash(
                    'success', new TranslatableMessage('import.flash_group_meeting_success', [
                    '%count%' => count($importedGroupMeetings),
                ]));
                $controller = in_array(User::ROLE_ADMIN, $this->getUser()->getRoles()) ? MeetingCrudController::class : MeetingCrudEducatorController::class;
                /** @var User $user */
                $user = $this->getUser();
                $targetUrl = $this->adminUrlGenerator
                    ->setDashboard($user->getDashboard())
                    ->setController($controller)
                    ->setAction(Crud::PAGE_INDEX)
                    ->generateUrl();
                return $this->redirect($targetUrl);
            }
            catch (TransformationFailedException $exception) {
                $this->addFlash(
                    'danger',
                    new TranslatableMessage($exception->getInvalidMessage(), $exception->getInvalidMessageParameters(), 'validators'),
                );
            }
            catch (Exception $exception) {
                $this->addFlash(
                    'danger',
                    $exception->getMessage()
                );
            }
        }
        return $this->render('basic_form.html.twig', [
            'form' => $form,
            'templateUrl' => $this->generateUrl('app_download_template_group_meeting')
        ]);
    }
    #[Route('/rozvrh', name: 'schedule')]
    #[IsGranted(User::ROLE_ADMIN)]
    public function schedule(Request $request, ImportSchedule $importSchedule): Response
    {
        $form = $this->createForm(ImportScheduleType::class);
        $form->handleRequest($request);
        if ($form->isSubmitted() && $form->isValid()) {
            $importedSchedules = $form->get('uploadedFile')->getData();
            $educator = $form->get('educator')->getData();
            try {
                $importSchedule->saveSchedule($importedSchedules, $educator);
                $this->addFlash(
                    'success', new TranslatableMessage('import.flash_schedule_success', [
                    '%count%' => count($importedSchedules),
                ]));
                $targetUrl = $this->adminUrlGenerator
                    ->setDashboard(AdminDashboardController::class)
                    ->setRoute('app_schedule_index')
                    ->generateUrl();
                return $this->redirect($targetUrl);
            }
            catch (TransformationFailedException $exception) {
                $this->addFlash(
                    'danger',
                    new TranslatableMessage($exception->getInvalidMessage(), $exception->getInvalidMessageParameters(), 'validators'),
                );
            }
            catch (Exception $exception) {
                $this->addFlash(
                    'danger',
                    $exception->getMessage()
                );
            }
        }
        return $this->render('basic_form.html.twig', [
            'form' => $form,
            'templateUrl' => $this->generateUrl('app_download_template_schedule')
        ]);
    }
    #[Route('/treningy', name: 'trainings')]
    public function trainings (Request $request, ImportTraining $importTraining): Response
    {
        $form = $this->createForm(ImportTrainingType::class);
        $form->handleRequest($request);
        if ($form->isSubmitted() && $form->isValid()) {
            $importedTraining = $form->get('uploadedFile')->getData();
            try {
                $importTraining->saveTraining($importedTraining);
                $this->addFlash(
                    'success','import.flash_training_success');
                $controller = in_array(User::ROLE_ADMIN, $this->getUser()->getRoles()) ? StudentCrudController::class : StudentCrudEducatorController::class;
                /** @var User $user */
                $user = $this->getUser();
                $targetUrl = $this->adminUrlGenerator
                    ->setController($controller)
                    ->setDashboard($user->getDashboard())
                    ->setAction(Crud::PAGE_INDEX)
                    ->generateUrl();
                return $this->redirect($targetUrl);
            }
            catch (TransformationFailedException $exception) {
                $this->addFlash(
                    'danger',
                    new TranslatableMessage($exception->getInvalidMessage(), $exception->getInvalidMessageParameters(), 'validators'),
                );
            }
            catch (Exception $exception) {
                $this->addFlash(
                    'danger',
                    $exception->getMessage(),
                );
            }
        }
        return $this->render('basic_form.html.twig', [
            'form' => $form,
            'templateUrl' => $this->generateUrl('app_download_template_trainings')
        ]);
    }
}
