<?php

namespace App\Controller;

use App\Entity\Message;
use App\Entity\MessageReceiver;
use App\Entity\User;
use App\Form\AnswerMessageType;
use App\Form\CreateMessageType;
use App\Repository\MessageRepository;
use App\Repository\UserRepository;
use App\Serializer\UserNormalizer;
use DateTimeImmutable;
use Doctrine\ORM\EntityManagerInterface;
use EasyCorp\Bundle\EasyAdminBundle\Router\AdminUrlGenerator;
use Exception;
use InvalidArgumentException;
use Knp\Component\Pager\PaginatorInterface;
use Symfony\Bundle\FrameworkBundle\Controller\AbstractController;
use Symfony\Component\HttpFoundation\JsonResponse;
use Symfony\Component\HttpFoundation\Request;
use Symfony\Component\HttpFoundation\Response;
use Symfony\Component\Routing\Annotation\Route;
use Symfony\Component\Security\Http\Attribute\IsGranted;
use Symfony\Component\Serializer\Encoder\JsonEncoder;
use Symfony\Component\Serializer\Serializer;
use Symfony\Component\Translation\TranslatableMessage;

#[Route('/sprava', name: 'app_message_')]
#[IsGranted('ROLE_USER')]
class MessageController extends AbstractController
{
    /**
     * @throws Exception
     */
    #[Route('/odpovedat', name: 'answer')]
    public function answer(Request $request, EntityManagerInterface $entityManager, MessageRepository $messageRepository,
                           UserRepository $userRepository, AdminUrlGenerator $adminUrlGenerator): Response
    {
        /** @var User $user */
        $user = $this->getUser();
        /** @var Message $message */
        $message = $messageRepository->findOneBy(['id' => $request->query->get('id')]);
        if ($message === null || !$message->isUserInMessageReceivers($user)) {
            throw $this->createNotFoundException();
        }
        $all = $request->query->get('all');
        if ($all !== '1' && $all !== '0') {
            throw $this->createNotFoundException();
        }
        if ($all === '1' && !($this->isGranted(User::ROLE_ADMIN) || $this->isGranted(User::ROLE_EDUCATOR))) {
            throw $this->createNotFoundException();
        }
        $form = $this->createForm(AnswerMessageType::class);
        $form->handleRequest($request);
        if ($form->isSubmitted() && $form->isValid()) {
            $text = $form->get('message')->getData();
            $newMessage = $this->createMessage($text);
            $newMessage->setSender($user);
            $newMessage->setParent($message);
            $newMessage->setRoot($message->getRoot());

            $receivers = [$message->getSender()];
            if ($all === '1') {
                $type = $message->getRoot()->getReceiverType();
                if ($type !== null) {
                    $receivers = array_merge($receivers, $userRepository->findReceiversByType($type, $user, $user->getEducator()));
                    $newMessage->setReceiverType($type);
                }
                else{
                    foreach ($message->getMessageReceivers() as $receiver) {
                        $receivers[] = $receiver->getReceiver();
                    }
                }

            }
            $receivers = array_unique($receivers);
            if (($key = array_search($user, $receivers)) !== false) {
                unset($receivers[$key]);
            }

            foreach ($receivers as $receiver) {
                $messageReceiver = $this->createMessageReceiver($receiver);
                $newMessage->addMessageReceiver($messageReceiver);
                $entityManager->persist($messageReceiver);
            }

            $entityManager->persist($newMessage);
            $entityManager->flush();
            if ($all === '0') {
                $this->addFlash('success', new TranslatableMessage('entity.message.flash_success_send_user', [
                    '%user%' => $message->getSender()
                ]));
            }
            else {
                $this->addFlash('success', new TranslatableMessage('entity.message.flash_success_group_send', [
                    '%group%' => $newMessage->getReceiverType() === null? 'N/A' : $newMessage->getReceiverType(),
                ]));
            }

            $url = $adminUrlGenerator
                ->setDashboard($user->getDashboard())
                ->setRoute('app_message_show_sent')
                ->generateUrl();
            return $this->redirect($url);
        }
        return $this->render('basic_form.html.twig', [
            'form' => $form,
            'templateUrl' => null,
        ]);
    }

    /**
     * @throws Exception
     */
    #[Route('/vytvorit', name: 'create')]
    public function create(Request $request, EntityManagerInterface $entityManager, UserRepository $userRepository, AdminUrlGenerator $adminUrlGenerator): Response
    {
        /** @var User $user */
        $user = $this->getUser();
        $form = $this->createForm(CreateMessageType::class, []);
        $form->handleRequest($request);
        if ($form->isSubmitted() && $form->isValid()) {
            $text = $form->get('message')->getData();
            $type = $form->get('receiverType')->getData();
            $receivers = $form->get('receivers')->getData();

            $newMessage = $this->createMessage($text);
            $newMessage->setSender($user);
            $entityManager->persist($newMessage);
            if ($receivers->isEmpty()) {
                // get receivers from type
                $receivers = $userRepository->findReceiversByType($type, $user, $user->getEducator());
                $newMessage->setReceiverType($type);
            }
            foreach ($receivers as $receiver) {
                $messageReceiver = $this->createMessageReceiver($receiver);
                $newMessage->addMessageReceiver($messageReceiver);
                $entityManager->persist($messageReceiver);
            }

            $entityManager->flush();
            $this->addFlash('success', 'entity.message.flash_success_send');

            $url = $adminUrlGenerator
                ->setDashboard($user->getDashboard())
                ->setRoute('app_message_show_sent')
                ->generateUrl();
            return $this->redirect($url);
        }
        return $this->render('message/create_message.html.twig', [
            'form' => $form,
        ]);
    }
    /**
     * @throws Exception
     */
    #[Route('/najst-prijimatelov', name: 'find_recipients')]
    public function findRecipients(Request $request, UserRepository $userRepository): Response
    {
        /** @var User $user */
        $user = $this->getUser();
        $type = $request->getContent();
        $recipients = $userRepository->findReceiversByType($type, $user, $user->getEducator());
        $serializer = new Serializer([new UserNormalizer()], [new JsonEncoder()]);
        $jsonContent = $serializer->serialize($recipients, 'json');
        return new JsonResponse($jsonContent);
    }

    /**
     * @throws Exception
     */
    #[Route('/ukaz-poslane', name: 'show_sent')]
    public function showSent(Request $request, MessageRepository $messageRepository, PaginatorInterface $paginator,
                             AdminUrlGenerator $adminUrlGenerator): Response
    {
        /** @var User $user */
        $user = $this->getUser();
        $buttonCreateUrl = $adminUrlGenerator
            ->setDashboard($user->getDashboard())
            ->setRoute('app_message_create')
            ->generateUrl();
        $buttonReceivedUrl = $adminUrlGenerator
            ->setDashboard($user->getDashboard())
            ->setRoute('app_message_show_received')
            ->generateUrl();

        $q = $request->query->get('query');
        $queryBuilder = $messageRepository->getWithSearchQueryBuilderSentMessages($q, $user);
        $pagination = $paginator->paginate(
            $queryBuilder, /* query NOT result */
            $request->query->getInt('page', 1)/*page number*/,
            10/*limit per page*/
        );

        return $this->render('message/show_sent_messages.html.twig', [
            'sentMessages' => $pagination,
            'buttonCreateUrl' => $buttonCreateUrl,
            'buttonReceivedUrl' => $buttonReceivedUrl,
        ]);
    }

    /**
     * @throws Exception
     */
    #[Route('/ukaz-prijate', name: 'show_received')]
    public function showReceived(Request $request, MessageRepository $messageRepository, PaginatorInterface $paginator, AdminUrlGenerator $adminUrlGenerator): Response
    {
        /** @var User $user */
        $user = $this->getUser();
        $buttonCreateUrl = $adminUrlGenerator
            ->setDashboard($user->getDashboard())
            ->setRoute('app_message_create')
            ->generateUrl();
        $buttonSentUrl = $adminUrlGenerator
            ->setDashboard($user->getDashboard())
            ->setRoute('app_message_show_sent')
            ->generateUrl();

        $q = $request->query->get('query');
        $queryBuilder = $messageRepository->getWithSearchQueryBuilderReceivedMessages($q, $user);
        $pagination = $paginator->paginate(
            $queryBuilder, /* query NOT result */
            $request->query->getInt('page', 1)/*page number*/,
            10/*limit per page*/
        );
        $unreadMessages = $messageRepository->findUnreadMessagesForUser($user);
        return $this->render('message/show_received_messages.html.twig', [
            'receivedMessages' => $pagination,
            'unreadMessages' => $unreadMessages,
            'buttonCreateUrl' => $buttonCreateUrl,
            'buttonSentUrl' => $buttonSentUrl,
        ]);
    }
    #[Route('/detail', name: 'detail')]
    public function detail(Request $request, MessageRepository $messageRepository, EntityManagerInterface $entityManager): Response
    {
        /** @var User $user */
        $user = $this->getUser();
        /** @var Message $message */
        $message = $messageRepository->findOneBy(['id' => $request->query->get('id')]);
        if ($message === null || (!$message->isUserInMessageReceivers($user) && $message->getSender() !== $user)) {
            throw $this->createNotFoundException();
        }
        $messageReceiver = $message->getMessageReceiverForGivenUser($user);
        if ($messageReceiver !== null) {
            $messageReceiver->setRead(true);
            $entityManager->flush();
        }
        $root = $message->getRoot();
        if ($root === null) {
            throw new InvalidArgumentException('Something went wrong');
        }
        $messageIds = $messageRepository->getMessagesIdByRoot($root);
        $messages = $messageRepository->getMessagesByIds($messageIds, $user);
        $conversation = $this->createConversation($messages);

        uksort($conversation, function($a, $b) {
            return strtotime($a) - strtotime($b);
        });
        return $this->render('message/detail_message.html.twig', [
            'conversation' => $conversation,
        ]);
    }
    private function createMessage(string $text): Message
    {
        $newMessage = new Message();
        $date = new DateTimeImmutable();

        $newMessage->setDate($date);
        $newMessage->setMessage($text);
        $newMessage->setParent(null);
        $newMessage->setRoot($newMessage);

        return $newMessage;
    }
    private function createMessageReceiver(User $receiver): MessageReceiver
    {
        $messageReceiver = new MessageReceiver();
        $messageReceiver->setReceiver($receiver);
        $messageReceiver->setRead(false);
        $messageReceiver->setSent(false);
        return $messageReceiver;
    }

    private function createConversation(array $messages): array
    {
        $conversation = [];
        foreach ($messages as $message) {
            $conversation[$message->getDate()->format('Y-m-d H:i:s')] = $message;
        }
        return $conversation;
    }
}
