<?php

namespace App\Controller;

use App\Controller\Admin\AdminDashboardController;
use App\Entity\Schedule;
use App\Entity\User;
use App\Form\EntityScheduleType;
use App\Form\FilterScheduleType;
use App\Repository\ScheduleRepository;
use App\Services\ScheduleService;
use Doctrine\ORM\EntityManagerInterface;
use EasyCorp\Bundle\EasyAdminBundle\Router\AdminUrlGenerator;
use Exception;
use Symfony\Bundle\FrameworkBundle\Controller\AbstractController;
use Symfony\Component\ExpressionLanguage\Expression;
use Symfony\Component\HttpFoundation\RedirectResponse;
use Symfony\Component\HttpFoundation\Request;
use Symfony\Component\HttpFoundation\Response;
use Symfony\Component\Routing\Annotation\Route;
use Symfony\Component\Security\Http\Attribute\IsGranted;

#[Route('/rozvrh', name: 'app_schedule_')]
#[IsGranted(new Expression('is_granted("ROLE_ADMIN") or is_granted("ROLE_EDUCATOR")'))]
class ScheduleController extends AbstractController
{
    /**
     * @throws Exception
     */
    #[Route('/ukaz', name: 'index')]
    public function index(int $app_starting_year, int $app_week_cycle, Request $request,
                          AdminUrlGenerator $adminUrlGenerator, ScheduleRepository $scheduleRepository): Response
    {
        /** @var User $user */
        $user = $this->getUser();
        $educator = $user->getEducator();
        $form = $this->createForm(FilterScheduleType::class,[], [
            'data' => [
                'educator' => $educator,
                'minYear' => $app_starting_year,
            ],
        ]);
        $schedules = [];
        $form->handleRequest($request);
        $buttonImportUrl = $adminUrlGenerator->setDashboard($user->getDashboard())->setRoute('app_import_schedule')->generateUrl();
        $buttonCreateUrl = $adminUrlGenerator->setDashboard($user->getDashboard())->setRoute('app_schedule_create')->generateUrl();
        $buttonDeleteAllUrl = $adminUrlGenerator->setDashboard($user->getDashboard())->setRoute('app_schedule_delete_all')->generateUrl();
        if ($form->isSubmitted() && $form->isValid()) {
            $week = $form->get('week')->getData();
            $educator = $form->get('educator')->getData();
            $schoolWeek = ScheduleService::findSchoolWeek($week);
            $weekOfCycle = ScheduleService::getWeekOfCycle($week, $schoolWeek, $app_week_cycle);
            $schedules = $scheduleRepository->findBy([
                'educator' => $educator,
                'week' => $weekOfCycle,
                ], ['week' => 'ASC', 'fromTime' => 'ASC', 'toTime' => 'ASC']);

            if (empty($schedules)) {
                $this->addFlash('warning', 'entity.schedule.flash_warning_no_data');
                return $this->render('schedule/show_schedule.html.twig', [
                    'form' => $form,
                    'schedules' => [],
                    'buttonImportUrl' => $buttonImportUrl,
                    'buttonCreateUrl' => $buttonCreateUrl,
                    'buttonDeleteAllUrl' => $buttonDeleteAllUrl,
                ]);
            }
            $schedules = $this->modifyByDay($schedules);
            $schedules = $this->addEmptyValues($schedules);
        }

        return $this->render('schedule/show_schedule.html.twig', [
            'form' => $form,
            'schedules' => $schedules,
            'buttonImportUrl' => $buttonImportUrl,
            'buttonCreateUrl' => $buttonCreateUrl,
            'buttonDeleteAllUrl' => $buttonDeleteAllUrl,
        ]);
    }
    #[Route('/novy_rozvrh', name: 'create')]
    #[IsGranted(User::ROLE_ADMIN)]
    public function create(Request $request, EntityManagerInterface $entityManager, AdminUrlGenerator $adminUrlGenerator): Response
    {
        /** @var User $user */
        $user = $this->getUser();
        $educator = $user->getEducator();
        $schedule = new Schedule();
        $schedule->setEducator($educator);
        return $this->showScheduleFormAndSave($schedule, $request, $entityManager, $adminUrlGenerator);
    }

    #[Route('/uprav_rozvrh/{id}', name: 'modify')]
    #[IsGranted(User::ROLE_ADMIN)]
    public function modify(Schedule $schedule, Request $request, EntityManagerInterface $entityManager, AdminUrlGenerator $adminUrlGenerator): Response
    {
        return $this->showScheduleFormAndSave($schedule, $request, $entityManager, $adminUrlGenerator);
    }
    /**
     * @param Schedule $schedule
     * @param Request $request
     * @param EntityManagerInterface $entityManager
     * @param AdminUrlGenerator $adminUrlGenerator
     * @return RedirectResponse|Response
     */
    public function showScheduleFormAndSave(Schedule $schedule, Request $request, EntityManagerInterface $entityManager, AdminUrlGenerator $adminUrlGenerator): Response|RedirectResponse
    {
        $form = $this->createForm(EntityScheduleType::class, $schedule);
        $form->handleRequest($request);
        if ($form->isSubmitted() && $form->isValid()) {
            $entityManager->persist($schedule);
            $entityManager->flush();
            $this->addFlash('success', 'entity.schedule.flash_saved_changes');
            $url = $adminUrlGenerator->setDashboard(AdminDashboardController::class)->setRoute('app_schedule_index')->generateUrl();
            return $this->redirect($url);
        }
        return $this->render('basic_form.html.twig', [
            'form' => $form,
            'templateUrl' => null,
        ]);
    }
    #[Route('/zmaz_rozvrh/{id}', name: 'delete')]
    #[IsGranted(User::ROLE_ADMIN)]
    public function delete(Schedule $schedule, Request $request, EntityManagerInterface $entityManager, AdminUrlGenerator $adminUrlGenerator): Response
    {
        $entityManager->remove($schedule);
        $entityManager->flush();
        $this->addFlash('success', 'entity.schedule.flash_success_delete');
        $url = $adminUrlGenerator->setDashboard(AdminDashboardController::class)->setRoute('app_schedule_index')->generateUrl();
        return $this->redirect($url);
    }

    #[Route('/zmaz-vsetky-rozvrhy', name: 'delete_all')]
    #[IsGranted(User::ROLE_ADMIN)]
    public function deleteAll(Request $request, EntityManagerInterface $entityManager, AdminUrlGenerator $adminUrlGenerator, ScheduleRepository $scheduleRepository): Response
    {
        $scheduleRepository->removeAllSchedules();
        $entityManager->flush();
        $this->addFlash('success', 'entity.schedule.flash_success_delete_all');
        $url = $adminUrlGenerator->setDashboard(AdminDashboardController::class)->setRoute('app_schedule_index')->generateUrl();
        return $this->redirect($url);
    }

    private function addEmptyValues(array $schedules): array
    {
        for ($i = 0; $i < 7; $i++) {
            $found = $this->checkIfExistsScheduleBasedOnDayOfWeek($schedules, $i);
            if ($found === false) {
                $schedules[$i] = [];
            }
        }
        // Move dayInWeek = 0 to the end
        uksort($schedules, function($a, $b) {
            if ($a == 0) {
                return 1;
            } elseif ($b == 0) {
                return -1;
            } else {
                return $a - $b;
            }
        });
        return $schedules;
    }
    private function checkIfExistsScheduleBasedOnDayOfWeek(array $schedules, int $i): bool
    {
        if (array_key_exists($i, $schedules)) {
            return true;
        }
        return false;
    }

    private function modifyByDay(array $schedules): array
    {
        $result = [];
        /** @var Schedule $schedule */
        foreach ($schedules as $schedule){
            $result[$schedule->getDayInWeek()][] = $schedule;
        }
        return $result;
    }
}
