<?php

namespace App\Controller\Student\Crud;

use App\Controller\Admin\Crud\MeetingCrudController;
use App\Entity\Meeting;
use App\Entity\User;
use Doctrine\ORM\QueryBuilder;
use EasyCorp\Bundle\EasyAdminBundle\Collection\FieldCollection;
use EasyCorp\Bundle\EasyAdminBundle\Collection\FilterCollection;
use EasyCorp\Bundle\EasyAdminBundle\Config\Action;
use EasyCorp\Bundle\EasyAdminBundle\Config\Actions;
use EasyCorp\Bundle\EasyAdminBundle\Config\Crud;
use EasyCorp\Bundle\EasyAdminBundle\Config\Filters;
use EasyCorp\Bundle\EasyAdminBundle\Dto\EntityDto;
use EasyCorp\Bundle\EasyAdminBundle\Dto\SearchDto;
use EasyCorp\Bundle\EasyAdminBundle\Field\AssociationField;
use EasyCorp\Bundle\EasyAdminBundle\Field\CollectionField;
use EasyCorp\Bundle\EasyAdminBundle\Field\DateTimeField;
use EasyCorp\Bundle\EasyAdminBundle\Field\FormField;
use EasyCorp\Bundle\EasyAdminBundle\Field\IdField;
use EasyCorp\Bundle\EasyAdminBundle\Field\IntegerField;
use EasyCorp\Bundle\EasyAdminBundle\Field\TextField;
use Symfony\Component\Security\Http\Attribute\IsGranted;

#[IsGranted(User::ROLE_STUDENT)]
class MeetingCrudStudentController extends MeetingCrudController
{
    public static function getEntityFqcn(): string
    {
        return Meeting::class;
    }

    public function configureActions(Actions $actions): Actions
    {
        return parent::configureActions($actions)
            ->remove(Crud::PAGE_DETAIL, Action::DELETE)
            ->remove(Crud::PAGE_INDEX, Action::DELETE)
            ->disable(Action::BATCH_DELETE, Action::DELETE)
            ;
    }
    public function configureFilters(Filters $filters): Filters
    {
        return $filters
            ->add('plannedDate')
            ;
    }
    public function createIndexQueryBuilder(SearchDto $searchDto, EntityDto $entityDto, FieldCollection $fields, FilterCollection $filters): QueryBuilder
    {
        /** @var User $user */
        $user = $this->getUser();
        return parent::createIndexQueryBuilder($searchDto, $entityDto, $fields, $filters)
            ->leftJoin('entity.meetingAttendances', 'meetingAttendances_student_index')
            ->andWhere(':student = meetingAttendances_student_index.student')
            ->setParameter('student', $user->getStudent()->getId());
    }
    public function configureFields(string $pageName): iterable
    {
        return [
            FormField::addTab('dashboard.general_info'),
            IdField::new('id')
                ->setLabel('entity.id')
                ->hideOnForm(),
            AssociationField::new('educator')
                ->setLabel('entity.educator.singular_name')
                ->setCrudController(EducatorCrudStudentController::class)
                ->setColumns('col-12')
                ->hideOnForm()
                ->autocomplete()
                ->setRequired(true),
            DateTimeField::new('plannedDate')
                ->setLabel('entity.meeting.planned_date')
                ->setColumns('col-12')
                ->setRequired(true),
            TextField::new('studentNote')
                ->setLabel('entity.meeting.student_note')
                ->setColumns('col-12')
                ->hideOnIndex(),
            IntegerField::new('doorNumber')
                ->setLabel('entity.meeting.door_number')
                ->setColumns('col-12')
                ->setRequired(true),
            FormField::addTab('entity.student.plural_name'),
            CollectionField::new('meetingAttendances')
                ->setLabel('entity.meeting.attendance')
                ->hideOnIndex()
                ->setTemplatePath('admin/show_meeting_students.html.twig'),
            TextField::new('type')
                ->setLabel('entity.meeting.type')
                ->setFormTypeOption('mapped', false)
                ->hideOnForm()
                ->hideOnDetail()
                ->setValue('N/A')
                ->formatValue(function ($value, $entity) {
                    return $this->individualOrGroupMeeting($entity);
                }),
        ];
    }
}