<?php

namespace App\Controller\Student;

use App\Controller\Student\Crud\EducatorCrudStudentController;
use App\Controller\Student\Crud\LeavingHomeCrudStudentController;
use App\Controller\Student\Crud\MeetingCrudStudentController;
use App\Controller\Student\Crud\WalkOutCrudStudentController;
use App\Entity\Attendance;
use App\Entity\Educator;
use App\Entity\Meeting;
use App\Entity\User;
use App\Repository\MessageReceiverRepository;
use App\Repository\ScheduleRepository;
use App\Services\ScheduleService;
use DateTimeImmutable;
use EasyCorp\Bundle\EasyAdminBundle\Config\Action;
use EasyCorp\Bundle\EasyAdminBundle\Config\Actions;
use EasyCorp\Bundle\EasyAdminBundle\Config\Assets;
use EasyCorp\Bundle\EasyAdminBundle\Config\Crud;
use EasyCorp\Bundle\EasyAdminBundle\Config\Dashboard;
use EasyCorp\Bundle\EasyAdminBundle\Config\MenuItem;
use EasyCorp\Bundle\EasyAdminBundle\Config\UserMenu;
use EasyCorp\Bundle\EasyAdminBundle\Controller\AbstractDashboardController;
use EasyCorp\Bundle\EasyAdminBundle\Router\AdminUrlGenerator;
use Exception;
use Symfony\Component\HttpFoundation\Response;
use Symfony\Component\Routing\Annotation\Route;
use Symfony\Component\Security\Core\User\UserInterface;
use Symfony\Component\Security\Http\Attribute\IsGranted;
use Symfony\Contracts\Translation\TranslatorInterface;

#[IsGranted('ROLE_STUDENT')]
class StudentDashboardController extends AbstractDashboardController
{
    private ScheduleRepository $scheduleRepository;
    private AdminUrlGenerator $adminUrlGenerator;
    private MessageReceiverRepository $messageReceiverRepository;
    private TranslatorInterface $translator;

    public function __construct(ScheduleRepository $scheduleRepository, AdminUrlGenerator $adminUrlGenerator, TranslatorInterface $translator,
                                MessageReceiverRepository $messageReceiverRepository)
    {
        $this->scheduleRepository = $scheduleRepository;
        $this->adminUrlGenerator = $adminUrlGenerator;
        $this->messageReceiverRepository = $messageReceiverRepository;
        $this->translator = $translator;
    }

    /**
     * @throws Exception
     */
    #[Route('/student', name: 'student')]
    public function index(): Response
    {
        $date = new DateTimeImmutable();
        $educatorUrl = $this->generateCrudUrl(EducatorCrudStudentController::class);
        $meetingUrl = $this->generateCrudUrl(MeetingCrudStudentController::class);
        $walkUrl = $this->generateCrudUrl(WalkOutCrudStudentController::class);
        $leaveUrl = $this->generateCrudUrl(LeavingHomeCrudStudentController::class);
        $messageUrl =  $this->adminUrlGenerator
            ->setDashboard(StudentDashboardController::class)
            ->setRoute('app_message_show_received')
            ->generateUrl();
        $attendanceUrl = $this->adminUrlGenerator
            ->setDashboard(StudentDashboardController::class)
            ->setRoute('app_attendance_view', ['id' => $this->getUser()->getStudent()->getId()])
            ->generateUrl();
        return $this->render('dashboard.html.twig', [
            'schedules' => $this->scheduleRepository->findScheduleForDate($date),
            'currentDate' => $date,
            'freeDay' => (new ScheduleService)->checkIfFreeDay($date),
            'unreadMessagesCount' => $this->messageReceiverRepository->count(['isRead' => false, 'receiver' => $this->getUser()]),
            'url' => [
                'educator' => $educatorUrl,
                'student' => null,
                'meeting' => $meetingUrl,
                'walk' => $walkUrl,
                'leave' => $leaveUrl,
                'note' => null,
                'message' => $messageUrl,
                'room' => null,
                'attendance' => $attendanceUrl,
            ]
        ]);
    }
    private function generateCrudUrl(string $class): string
    {
        return $this->adminUrlGenerator
            ->setDashboard(StudentDashboardController::class)
            ->setController($class)
            ->setAction('index')
            ->generateUrl();
    }


    public function configureDashboard(): Dashboard
    {
        return Dashboard::new()
            ->setTitle($this->translator->trans('dashboard.main_name'));
    }
    public function configureCrud(): Crud
    {
        return Crud::new()
            // the max number of entities to display per page
            ->setPaginatorPageSize(30)
            // the number of pages to display on each side of the current page
            // e.g. if num pages = 35, current page = 7 and you set ->setPaginatorRangeSize(4)
            // the paginator displays: [Previous]  1 ... 3  4  5  6  [7]  8  9  10  11 ... 35  [Next]
            // set this number to 0 to display a simple "< Previous | Next >" pager
            ->setPaginatorRangeSize(4)
            ;
    }
    public function configureAssets(): Assets
    {
        return parent::configureAssets()
            ->addWebpackEncoreEntry('easyadmin');
    }
    public function configureActions(): Actions
    {
        return parent::configureActions()
            ->add(Crud::PAGE_INDEX, Action::DETAIL)
            ->update(Crud::PAGE_DETAIL, Action::EDIT, static function (Action $action) {
                return $action->setIcon('fa fa-edit');
            })
            ->update(Crud::PAGE_DETAIL, Action::INDEX, static function (Action $action) {
                return $action->setIcon('fa fa-list');
            })
            ;
    }
    public function configureMenuItems(): iterable
    {
        yield MenuItem::linkToDashboard('dashboard.main_panel', 'fa fa-home');
        yield MenuItem::linkToRoute('entity.message.plural_name', 'fas fa-envelope', 'app_message_show_received');
        yield MenuItem::linkToCrud('entity.educator.plural_name', 'fas fa-user-tie', Educator::class)
            ->setController(EducatorCrudStudentController::class);
        yield MenuItem::linkToRoute('entity.attendance.singular_name', 'fas fa-clock', 'app_attendance_view', ['id' => $this->getUser()->getStudent()->getId()]);
        yield MenuItem::linkToCrud('entity.meeting.plural_name', 'fas fa-handshake-simple', Meeting::class)
            ->setController(MeetingCrudStudentController::class);
        yield MenuItem::linkToCrud('entity.attendance.action.walk_out.plural_name', 'fas fa-shoe-prints', Attendance::class)
            ->setController(WalkOutCrudStudentController::class);
        yield MenuItem::linkToCrud('entity.attendance.action.departure.plural_name', 'fas fa-paper-plane', Attendance::class)
            ->setController(LeavingHomeCrudStudentController::class);
    }
    public function configureUserMenu(UserInterface $user): UserMenu
    {
        /** @var User $user */
        return parent::configureUserMenu($user)
            ->setName($user->getFirstName() . ' ' . $user->getLastName() . ' ' . $user->getEmail())
            ->addMenuItems([
                MenuItem::linkToRoute('entity.user.profile', 'fa fa-id-card', 'app_user_profile'),
            ]);
    }
}
