<?php

namespace App\Controller;

use App\Entity\User;
use App\Services\UserSubscriptionManager;
use Symfony\Bundle\FrameworkBundle\Controller\AbstractController;
use Symfony\Component\HttpFoundation\Request;
use Symfony\Component\HttpFoundation\Response;
use Symfony\Component\HttpKernel\Exception\AccessDeniedHttpException;
use Symfony\Component\HttpKernel\Exception\BadRequestHttpException;
use Symfony\Component\HttpKernel\Exception\MethodNotAllowedHttpException;
use Symfony\Component\Routing\Annotation\Route;
use Symfony\Component\Security\Core\User\UserInterface;
use Symfony\Component\Security\Http\Attribute\IsGranted;

#[Route('/subscribe', name: 'app_subscribe_')]
#[IsGranted('ROLE_USER')]
class SubscribeController extends AbstractController
{

    #[Route('/save', name: 'save', methods: ['POST', 'DELETE'])]
    public function save(Request $request, UserSubscriptionManager $userSubscriptionManager): Response
    {
        /** @var User $user */
        $user = $this->getUser();
        if (null === $user) {
            throw new AccessDeniedHttpException('Not authenticated.');
        }
        if (!in_array($request->getMethod(), ['POST', 'DELETE'])) {
            throw new MethodNotAllowedHttpException(['POST', 'DELETE']);
        }
        $data = json_decode($request->getContent(), true);
        $subscription = $data['subscription'] ?? [];
        $options = $data['options'] ?? [];

        if (JSON_ERROR_NONE !== json_last_error()) {
            throw new BadRequestHttpException(json_last_error_msg());
        }

        if (!isset($subscription['endpoint'])) {
            throw new BadRequestHttpException('Invalid subscription object.');
        }

        $subscriptionHash = $userSubscriptionManager->hash($subscription['endpoint'], $user);
        if ('DELETE' === $request->getMethod()) {
            $this->unsubscribe($userSubscriptionManager, $user, $subscriptionHash);
        } else {
            $this->subscribe($userSubscriptionManager, $user, $subscriptionHash, $subscription, $options);
        }

        return new Response('', Response::HTTP_NO_CONTENT);
    }

    private function subscribe(UserSubscriptionManager $userSubscriptionManager, User $user, string $subscriptionHash, array $subscription, array $options = []): void
    {
        $userSubscription = $userSubscriptionManager->getUserSubscription($user, $subscriptionHash)
        or $userSubscription = $userSubscriptionManager->factory($user, $subscriptionHash, $subscription, $options);
        $userSubscriptionManager->save($userSubscription);
    }
    private function unsubscribe(UserSubscriptionManager $userSubscriptionManager, User $user, string $subscriptionHash): void
    {
        $subscription = $userSubscriptionManager->getUserSubscription($user, $subscriptionHash);
        if (null === $subscription) {
            throw new BadRequestHttpException('Subscription hash not found');
        }
        $userSubscriptionManager->delete($subscription);
    }
}
