<?php

namespace App\Entity;

use App\Repository\AttendanceRepository;
use DateTimeImmutable;
use Doctrine\DBAL\Types\Types;
use Doctrine\ORM\Mapping as ORM;
use Exception;
use App\Validator\Constraints as CustomAssert;
use Symfony\Component\Validator\Constraints as Assert;



#[ORM\Entity(repositoryClass: AttendanceRepository::class)]
#[ORM\Index(fields: ['time', 'student'])]
#[CustomAssert\WalkValidUntilSameDayAsTime]
#[CustomAssert\LeaveValidUntilDifferentDayAsTime]
class Attendance
{
    public const PRESENT_ACTION = 'present';
    public const ABSENT_ACTION = 'absent';
    const MONDAY = 1;
    const TUESDAY = 2;
    const WEDNESDAY = 3;
    const THURSDAY = 4;
    const FRIDAY = 5;
    const SATURDAY = 6;
    const SUNDAY = 0;

    public const PRESENT = 0;
    public const ABSENT = 1;
    public const EVENING_CHECK = 2;

    public const WALK = 4; //from-to

    public const DEPARTURE = 5; //from
    #[ORM\Id]
    #[ORM\GeneratedValue]
    #[ORM\Column]
    private ?int $id = null;

    #[ORM\ManyToOne(inversedBy: 'attendances')]
    #[ORM\JoinColumn(nullable: false)]
    #[Assert\NotNull()]
    private ?Student $student = null;

    #[ORM\Column]
    private ?DateTimeImmutable $time = null;

    #[ORM\Column(nullable: true)]
    private ?DateTimeImmutable $validUntil = null;

    #[ORM\Column(type: Types::SMALLINT)]
    private ?int $action = null;

    #[ORM\Column(length: 255, nullable: true)]
    private ?string $note = null;


    /**
     * @throws Exception
     */
    public function getActionToText(): string
    {
        switch ($this->action) {
            case Attendance::PRESENT: {
                return 'entity.attendance.action.present.singular_name';
            }
            case Attendance::ABSENT: {
                return 'entity.attendance.action.absent.singular_name';
            }
            case Attendance::EVENING_CHECK: {
                return 'entity.attendance.evening_control';
            }
            case Attendance::WALK: {
                return 'entity.attendance.action.walk_out.singular_name';
            }
            case Attendance::DEPARTURE: {
                return 'entity.attendance.action.departure.singular_name';
            }
            default:
                throw new Exception('unknown action');
        }
    }
    public function getId(): ?int
    {
        return $this->id;
    }

    public function getStudent(): ?Student
    {
        return $this->student;
    }

    public function setStudent(?Student $student): static
    {
        $this->student = $student;

        return $this;
    }

    public function getTime(): ?DateTimeImmutable
    {
        return $this->time;
    }

    public function setTime(DateTimeImmutable $time): static
    {
        $this->time = $time;

        return $this;
    }

    public function getValidUntil(): ?DateTimeImmutable
    {
        return $this->validUntil;
    }

    public function setValidUntil(?DateTimeImmutable $validUntil): static
    {
        $this->validUntil = $validUntil;

        return $this;
    }

    public function getAction(): ?int
    {
        return $this->action;
    }

    public function setAction(int $action): static
    {
        $this->action = $action;

        return $this;
    }

    public function getNote(): ?string
    {
        return $this->note;
    }

    public function setNote(?string $note): static
    {
        $this->note = $note;

        return $this;
    }
}
