<?php

namespace App\Entity;

use App\Repository\EducatorRepository;
use Doctrine\Common\Collections\ArrayCollection;
use Doctrine\Common\Collections\Collection;
use Doctrine\ORM\Mapping as ORM;
use Symfony\Component\Validator\Constraints as Assert;

#[ORM\Entity(repositoryClass: EducatorRepository::class)]
class Educator
{
    #[ORM\Id]
    #[ORM\GeneratedValue]
    #[ORM\Column]
    private ?int $id = null;

    #[ORM\OneToMany(mappedBy: 'educator', targetEntity: Student::class)]
    private Collection $students;

    #[ORM\OneToMany(mappedBy: 'educator', targetEntity: Meeting::class)]
    private Collection $meetings;

    #[ORM\OneToMany(mappedBy: 'educator', targetEntity: Schedule::class)]
    private Collection $schedules;

    #[ORM\ManyToOne(cascade: ['persist'], inversedBy: 'educators')]
    #[ORM\JoinColumn(nullable: false)]
    #[Assert\Valid]
    private ?User $user = null;

    #[ORM\OneToMany(mappedBy: 'educator', targetEntity: StudentNote::class)]
    private Collection $studentNotes;

    public function __construct()
    {
        $this->students = new ArrayCollection();
        $this->meetings = new ArrayCollection();
        $this->schedules = new ArrayCollection();
        $this->studentNotes = new ArrayCollection();
    }

    public function __toString(): string
    {
        return $this->user->getFirstName() . ' ' . $this->user->getLastName(). ' ' .$this->user->getEmail();
    }

    public function getId(): ?int
    {
        return $this->id;
    }

    public function getUser(): ?User
    {
        return $this->user;
    }

    public function setUser(?User $user): static
    {
        $this->user = $user;

        return $this;
    }

    /**
     * @return Collection<int, Student>
     */
    public function getStudents(): Collection
    {
        return $this->students;
    }

    public function addStudent(Student $student): static
    {
        if (!$this->students->contains($student)) {
            $this->students->add($student);
            $student->setEducator($this);
        }

        return $this;
    }

    public function removeStudent(Student $student): static
    {
        if ($this->students->removeElement($student)) {
            // set the owning side to null (unless already changed)
            if ($student->getEducator() === $this) {
                $student->setEducator(null);
            }
        }

        return $this;
    }

    /**
     * @return Collection<int, Meeting>
     */
    public function getMeetings(): Collection
    {
        return $this->meetings;
    }

    public function addMeeting(Meeting $meeting): static
    {
        if (!$this->meetings->contains($meeting)) {
            $this->meetings->add($meeting);
            $meeting->setEducator($this);
        }

        return $this;
    }

    public function removeMeeting(Meeting $meeting): static
    {
        if ($this->meetings->removeElement($meeting)) {
            // set the owning side to null (unless already changed)
            if ($meeting->getEducator() === $this) {
                $meeting->setEducator(null);
            }
        }

        return $this;
    }

    /**
     * @return Collection<int, Schedule>
     */
    public function getSchedules(): Collection
    {
        return $this->schedules;
    }

    public function addSchedule(Schedule $schedule): static
    {
        if (!$this->schedules->contains($schedule)) {
            $this->schedules->add($schedule);
            $schedule->setEducator($this);
        }

        return $this;
    }

    public function removeSchedule(Schedule $schedule): static
    {
        if ($this->schedules->removeElement($schedule)) {
            // set the owning side to null (unless already changed)
            if ($schedule->getEducator() === $this) {
                $schedule->setEducator(null);
            }
        }

        return $this;
    }

    /**
     * @return Collection<int, StudentNote>
     */
    public function getStudentNotes(): Collection
    {
        return $this->studentNotes;
    }

    public function addStudentNote(StudentNote $studentNote): static
    {
        if (!$this->studentNotes->contains($studentNote)) {
            $this->studentNotes->add($studentNote);
            $studentNote->setEducator($this);
        }

        return $this;
    }

    public function removeStudentNote(StudentNote $studentNote): static
    {
        if ($this->studentNotes->removeElement($studentNote)) {
            // set the owning side to null (unless already changed)
            if ($studentNote->getEducator() === $this) {
                $studentNote->setEducator(null);
            }
        }

        return $this;
    }
}
