<?php

namespace App\Entity;

use App\Repository\MessageRepository;
use DateTimeImmutable;
use Doctrine\Common\Collections\ArrayCollection;
use Doctrine\Common\Collections\Collection;
use Doctrine\DBAL\Types\Types;
use Doctrine\ORM\Mapping as ORM;
use PhpOffice\PhpSpreadsheet\Shared\OLE\PPS\Root;

#[ORM\Entity(repositoryClass: MessageRepository::class)]
class Message
{
    public const ALL_TYPE = 'all';
    public const EDUCATOR_TYPE = 'educators';
    public const STUDENT_TYPE = 'students';
    public const GUARDIAN_TYPE = 'guardians';
    public const STUDENT_AND_GUARDIAN_TYPE = 'students_and_guardians';
    public const EDUCATOR_STUDENTS_TYPE = 'educator\'s_students';

    #[ORM\Id]
    #[ORM\GeneratedValue]
    #[ORM\Column]
    private ?int $id = null;

    #[ORM\ManyToOne(inversedBy: 'sentMessages')]
    #[ORM\JoinColumn(nullable: false)]
    private ?User $sender = null;

    #[ORM\Column]
    private ?DateTimeImmutable $date = null;

    #[ORM\Column(type: Types::TEXT)]
    private ?string $message = null;

    #[ORM\ManyToOne(targetEntity: self::class, inversedBy: 'children')]
    private ?self $parent = null;

    #[ORM\OneToMany(mappedBy: 'parent', targetEntity: self::class)]
    private Collection $children;

    #[ORM\OneToMany(mappedBy: 'message', targetEntity: MessageReceiver::class, orphanRemoval: true)]
    private Collection $messageReceivers;

    #[ORM\Column(length: 255, nullable: true)]
    private ?string $receiverType = null;

    #[ORM\ManyToOne(targetEntity: self::class, inversedBy: 'descendants')]
    private ?self $root = null;

    #[ORM\OneToMany(mappedBy: 'root', targetEntity: self::class)]
    private Collection $descendants;
    public function isUserInMessageReceivers(User $user): bool
    {
        /** @var MessageReceiver $messageReceiver */
        foreach ($this->messageReceivers as $messageReceiver) {
            if ($user === $messageReceiver->getReceiver()) {
                return true;
            }
        }
        return false;
    }
    public function getMessageReceiverForGivenUser(User $user): ?MessageReceiver
    {
        /** @var MessageReceiver $messageReceiver */
        foreach ($this->messageReceivers as $messageReceiver) {
            if ($user === $messageReceiver->getReceiver()) {
                return $messageReceiver;
            }
        }
        return null;
    }
    public function __toString(): string
    {
        return 'Sprava od: ' . $this->getSender();
    }

    public function __construct()
    {
        $this->children = new ArrayCollection();
        $this->messageReceivers = new ArrayCollection();
        $this->descendants = new ArrayCollection();
    }

    public function getId(): ?int
    {
        return $this->id;
    }

    public function getSender(): ?User
    {
        return $this->sender;
    }

    public function setSender(?User $sender): static
    {
        $this->sender = $sender;

        return $this;
    }

    public function getDate(): ?DateTimeImmutable
    {
        return $this->date;
    }

    public function setDate(DateTimeImmutable $date): static
    {
        $this->date = $date;

        return $this;
    }

    public function getMessage(): ?string
    {
        return $this->message;
    }

    public function setMessage(string $message): static
    {
        $this->message = $message;

        return $this;
    }

    public function getParent(): ?self
    {
        return $this->parent;
    }

    public function setParent(?self $parent): static
    {
        $this->parent = $parent;

        return $this;
    }

    /**
     * @return Collection<int, self>
     */
    public function getChildren(): Collection
    {
        return $this->children;
    }

    public function addChild(self $child): static
    {
        if (!$this->children->contains($child)) {
            $this->children->add($child);
            $child->setParent($this);
        }
        return $this;
    }

    public function removeChild(self $child): static
    {
        if ($this->children->removeElement($child)) {
            // set the owning side to null (unless already changed)
            if ($child->getParent() === $this) {
                $child->setParent(null);
            }
        }
        return $this;
    }

    /**
     * @return Collection<int, MessageReceiver>
     */
    public function getMessageReceivers(): Collection
    {
        return $this->messageReceivers;
    }

    public function addMessageReceiver(MessageReceiver $messageReceiver): static
    {
        if (!$this->messageReceivers->contains($messageReceiver)) {
            $this->messageReceivers->add($messageReceiver);
            $messageReceiver->setMessage($this);
        }

        return $this;
    }

    public function removeMessageReceiver(MessageReceiver $messageReceiver): static
    {
        if ($this->messageReceivers->removeElement($messageReceiver)) {
            // set the owning side to null (unless already changed)
            if ($messageReceiver->getMessage() === $this) {
                $messageReceiver->setMessage(null);
            }
        }

        return $this;
    }

    public function getReceiverType(): ?string
    {
        return $this->receiverType;
    }

    public function setReceiverType(?string $receiverType): static
    {
        $this->receiverType = $receiverType;

        return $this;
    }

    public function getRoot(): ?self
    {
        return $this->root;
    }

    public function setRoot(?self $root): static
    {
        $this->root = $root;

        return $this;
    }

    /**
     * @return Collection<int, self>
     */
    public function getDescendants(): Collection
    {
        return $this->descendants;
    }

    public function addDescendant(self $descendant): static
    {
        if (!$this->descendants->contains($descendant)) {
            $this->descendants->add($descendant);
            $descendant->setRoot($this);
        }

        return $this;
    }

    public function removeDescendant(self $descendant): static
    {
        if ($this->descendants->removeElement($descendant)) {
            // set the owning side to null (unless already changed)
            if ($descendant->getRoot() === $this) {
                $descendant->setRoot(null);
            }
        }

        return $this;
    }
}
