<?php

namespace App\Entity;

use App\Repository\StudentRepository;
use App\Services\DailyAttendance;
use DateTimeImmutable;
use Doctrine\Common\Collections\ArrayCollection;
use Doctrine\Common\Collections\Collection;
use Doctrine\DBAL\Types\Types;
use Doctrine\ORM\Mapping as ORM;
use Symfony\Component\Validator\Constraints as Assert;
use App\Validator\Constraints as CustomAssert;


#[ORM\Entity(repositoryClass: StudentRepository::class)]
#[CustomAssert\TwoStudentsInOneRoom]
#[CustomAssert\NonActiveStudentNoRoom]
class Student
{
    #[ORM\Id]
    #[ORM\GeneratedValue]
    #[ORM\Column]
    private ?int $id = null;

    #[ORM\Column(length: 255, nullable: true)]
    private ?string $street = null;

    #[ORM\Column(length: 255, nullable: true)]
    private ?string $postalCode = null;

    #[ORM\Column(length: 255, nullable: true)]
    private ?string $city = null;

    #[ORM\Column(length: 2, nullable: true)]
    private ?string $country = null;

    #[ORM\ManyToOne(inversedBy: 'students')]
    #[Assert\NotBlank()]
    private ?Educator $educator = null;

    #[ORM\ManyToMany(targetEntity: Coach::class, mappedBy: 'students')]
    private Collection $coaches;

    #[ORM\ManyToOne(targetEntity: Room::class, inversedBy: 'students')]
    private ?Room $room = null;

    #[ORM\ManyToOne(inversedBy: 'students')]
    #[ORM\JoinColumn(nullable: true, onDelete: 'SET NULL')]
    private ?Classroom $classroom = null;

    #[ORM\OneToMany(mappedBy: 'student', targetEntity: Attendance::class, orphanRemoval: true)]
    private Collection $attendances;

    #[ORM\OneToMany(mappedBy: 'student', targetEntity: MeetingAttendance::class)]
    private Collection $meetingAttendances;

    #[ORM\Column(type: Types::DATE_IMMUTABLE, nullable: true)]
    private ?DateTimeImmutable $dateOfBirth = null;

    #[ORM\ManyToOne(targetEntity: MasterTeacher::class, inversedBy: 'students')]
    #[ORM\JoinColumn(nullable: true, onDelete: 'SET NULL')]
    private ?MasterTeacher $masterTeacher = null;

    #[ORM\ManyToOne(cascade: ['persist'], inversedBy: 'students')]
    #[ORM\JoinColumn(nullable: false)]
    #[Assert\Valid]
    private ?User $user = null;

    #[ORM\ManyToMany(targetEntity: Guardian::class, mappedBy: 'children')]
    private Collection $guardians;
    private array $dailyAttendances = [];

    #[ORM\ManyToMany(targetEntity: Activity::class, mappedBy: 'students')]
    private Collection $activities;

    #[ORM\OneToMany(mappedBy: 'student', targetEntity: StudentNote::class)]
    private Collection $studentNotes;

    #[ORM\OneToMany(mappedBy: 'student', targetEntity: DormitoryCard::class)]
    #[Assert\Count(max: 1)]
    private Collection $dormitoryCards;

    public function __construct()
    {
        $this->coaches = new ArrayCollection();
        $this->attendances = new ArrayCollection();
        $this->meetingAttendances = new ArrayCollection();
        $this->guardians = new ArrayCollection();
        $this->activities = new ArrayCollection();
        $this->studentNotes = new ArrayCollection();
        $this->dormitoryCards = new ArrayCollection();
    }
    public function __toString(): string
    {
        return $this->user->getFirstName() . ' ' . $this->user->getLastName(). ' ' .$this->user->getEmail();
    }
    public function getId(): ?int
    {
        return $this->id;
    }

    public function getStreet(): ?string
    {
        return $this->street;
    }

    public function setStreet(?string $street): static
    {
        $this->street = $street;

        return $this;
    }

    public function getPostalCode(): ?string
    {
        return $this->postalCode;
    }

    public function setPostalCode(?string $postalCode): static
    {
        $this->postalCode = $postalCode;

        return $this;
    }

    public function getCity(): ?string
    {
        return $this->city;
    }

    public function setCity(?string $city): static
    {
        $this->city = $city;

        return $this;
    }

    public function getCountry(): ?string
    {
        return $this->country;
    }

    public function setCountry(?string $country): static
    {
        $this->country = $country;

        return $this;
    }

    public function getEducator(): ?Educator
    {
        return $this->educator;
    }

    public function setEducator(?Educator $educator): static
    {
        $this->educator = $educator;

        return $this;
    }

    /**
     * @return Collection<int, Coach>
     */
    public function getCoaches(): Collection
    {
        return $this->coaches;
    }

    public function addCoach(Coach $coach): static
    {
        if (!$this->coaches->contains($coach)) {
            $this->coaches->add($coach);
            $coach->addStudent($this);
        }

        return $this;
    }

    public function removeCoach(Coach $coach): static
    {
        if ($this->coaches->removeElement($coach)) {
            $coach->removeStudent($this);
        }

        return $this;
    }

    public function getRoom(): ?Room
    {
        return $this->room;
    }

    public function setRoom(?Room $room): static
    {
        $this->room = $room;

        return $this;
    }

    public function getUser(): ?User
    {
        return $this->user;
    }

    public function setUser(?User $user): static
    {
        $this->user = $user;

        return $this;
    }

    public function getClassroom(): ?Classroom
    {
        return $this->classroom;
    }

    public function setClassroom(?Classroom $classroom): static
    {
        $this->classroom = $classroom;

        return $this;
    }

    /**
     * @return Collection<int, Attendance>
     */
    public function getAttendances(): Collection
    {
        return $this->attendances;
    }

    public function addAttendance(Attendance $attendance): static
    {
        if (!$this->attendances->contains($attendance)) {
            $this->attendances->add($attendance);
            $attendance->setStudent($this);
        }

        return $this;
    }

    public function removeAttendance(Attendance $attendance): static
    {
        if ($this->attendances->removeElement($attendance)) {
            // set the owning side to null (unless already changed)
            if ($attendance->getStudent() === $this) {
                $attendance->setStudent(null);
            }
        }

        return $this;
    }

    /**
     * @return Collection<int, MeetingAttendance>
     */
    public function getMeetingAttendances(): Collection
    {
        return $this->meetingAttendances;
    }

    public function addMeetingAttendance(MeetingAttendance $meetingAttendance): static
    {
        if (!$this->meetingAttendances->contains($meetingAttendance)) {
            $this->meetingAttendances->add($meetingAttendance);
            $meetingAttendance->setStudent($this);
        }

        return $this;
    }

    public function removeMeetingAttendance(MeetingAttendance $meetingAttendance): static
    {
        if ($this->meetingAttendances->removeElement($meetingAttendance)) {
            // set the owning side to null (unless already changed)
            if ($meetingAttendance->getStudent() === $this) {
                $meetingAttendance->setStudent(null);
            }
        }

        return $this;
    }

    public function getDateOfBirth(): ?DateTimeImmutable
    {
        return $this->dateOfBirth;
    }

    public function setDateOfBirth(?DateTimeImmutable $dateOfBirth): static
    {
        $this->dateOfBirth = $dateOfBirth;

        return $this;
    }

    public function getMasterTeacher(): ?MasterTeacher
    {
        return $this->masterTeacher;
    }

    public function setMasterTeacher(?MasterTeacher $masterTeacher): static
    {
        $this->masterTeacher = $masterTeacher;

        return $this;
    }

    /**
     * @return Collection<int, Guardian>
     */
    public function getGuardians(): Collection
    {
        return $this->guardians;
    }

    public function addGuardian(Guardian $guardian): static
    {
        if (!$this->guardians->contains($guardian)) {
            $this->guardians->add($guardian);
            $guardian->addChild($this);
        }

        return $this;
    }

    public function removeGuardian(Guardian $guardian): static
    {
        if ($this->guardians->removeElement($guardian)) {
            $guardian->removeChild($this);
        }

        return $this;
    }

    public function getDailyAttendance(DateTimeImmutable $date): DailyAttendance
    {
        if (array_key_exists($date->format('Y-m-d'), $this->dailyAttendances)) {
            return $this->dailyAttendances[$date->format('Y-m-d')];
        }

        $dailyAttendance = new DailyAttendance($date);
        $this->addDailyAttendance($dailyAttendance);

        return $dailyAttendance;
    }
    public function getDailyAttendances(): array
    {
        return $this->dailyAttendances;
    }

    public function addDailyAttendance(DailyAttendance $dailyAttendances): void
    {
        $this->dailyAttendances[$dailyAttendances->getDay()->format('Y-m-d')] = $dailyAttendances;
    }

    /**
     * @return Collection<int, Activity>
     */
    public function getActivities(): Collection
    {
        return $this->activities;
    }

    public function addActivity(Activity $activity): static
    {
        if (!$this->activities->contains($activity)) {
            $this->activities->add($activity);
            $activity->addStudent($this);
        }

        return $this;
    }

    public function removeActivity(Activity $activity): static
    {
        if ($this->activities->removeElement($activity)) {
            $activity->removeStudent($this);
        }

        return $this;
    }

    /**
     * @return Collection<int, StudentNote>
     */
    public function getStudentNotes(): Collection
    {
        return $this->studentNotes;
    }

    public function addStudentNote(StudentNote $studentNote): static
    {
        if (!$this->studentNotes->contains($studentNote)) {
            $this->studentNotes->add($studentNote);
            $studentNote->setStudent($this);
        }

        return $this;
    }

    public function removeStudentNote(StudentNote $studentNote): static
    {
        if ($this->studentNotes->removeElement($studentNote)) {
            // set the owning side to null (unless already changed)
            if ($studentNote->getStudent() === $this) {
                $studentNote->setStudent(null);
            }
        }

        return $this;
    }

    /**
     * @return Collection<int, DormitoryCard>
     */
    public function getDormitoryCards(): Collection
    {
        return $this->dormitoryCards;
    }

    public function addDormitoryCard(DormitoryCard $dormitoryCard): static
    {
        if (!$this->dormitoryCards->contains($dormitoryCard)) {
            $this->dormitoryCards->add($dormitoryCard);
            $dormitoryCard->setStudent($this);
        }

        return $this;
    }

    public function removeDormitoryCard(DormitoryCard $dormitoryCard): static
    {
        if ($this->dormitoryCards->removeElement($dormitoryCard)) {
            // set the owning side to null (unless already changed)
            if ($dormitoryCard->getStudent() === $this) {
                $dormitoryCard->setStudent(null);
            }
        }

        return $this;
    }
}
