<?php

namespace App\Entity;

use App\Controller\Admin\AdminDashboardController;
use App\Controller\Educator\EducatorDashboardController;
use App\Controller\Guardian\GuardianDashboardController;
use App\Controller\Student\StudentDashboardController;
use App\Repository\UserRepository;
use Doctrine\Common\Collections\ArrayCollection;
use Doctrine\Common\Collections\Collection;
use Doctrine\ORM\Mapping as ORM;
use Exception;
use Symfony\Bridge\Doctrine\Validator\Constraints\UniqueEntity;
use Symfony\Component\Security\Core\User\PasswordAuthenticatedUserInterface;
use Symfony\Component\Security\Core\User\UserInterface;
use App\Validator\Constraints as CustomAssert;
use Symfony\Component\Validator\Constraints as Assert;

#[ORM\Entity(repositoryClass: UserRepository::class)]
#[CustomAssert\E164Number]
#[UniqueEntity('email')]
class User implements UserInterface, PasswordAuthenticatedUserInterface
{
    public const ROLE_ADMIN = 'ROLE_ADMIN';
    public const ROLE_EDUCATOR = 'ROLE_EDUCATOR';
    public const ROLE_GUARDIAN = 'ROLE_GUARDIAN';
    public const ROLE_STUDENT = 'ROLE_STUDENT';
    public const ROLE_USER = 'ROLE_USER';

    #[ORM\Id]
    #[ORM\GeneratedValue]
    #[ORM\Column]
    private ?int $id = null;

    #[ORM\Column(length: 180, unique: true)]
    #[Assert\NotBlank]
    #[Assert\Email]
    private ?string $email = null;

    #[ORM\Column]
    private array $roles = [];

    /**
     * @var string The hashed password
     */
    #[ORM\Column]
    private string $password = 'NOT SET';
    #[ORM\Column]
    private ?bool $active = true;

    #[ORM\Column(length: 255)]
    #[Assert\NotBlank]
    private ?string $firstName = null;

    #[ORM\Column(length: 255)]
    #[Assert\NotBlank]
    private ?string $lastName = null;

    #[ORM\Column(length: 2, nullable: true)]
    private ?string $phoneCountry = null;

    #[ORM\Column(length: 16, nullable: true)]
    private ?string $phoneNumber = null;

    #[ORM\OneToMany(mappedBy: 'user', targetEntity: Student::class, orphanRemoval: true)]
    private Collection $students;

    #[ORM\OneToMany(mappedBy: 'user', targetEntity: Educator::class, orphanRemoval: true)]
    private Collection $educators;

    #[ORM\OneToMany(mappedBy: 'user', targetEntity: Guardian::class, orphanRemoval: true)]
    private Collection $guardians;

    #[ORM\OneToMany(mappedBy: 'user', targetEntity: Coach::class, orphanRemoval: true)]
    private Collection $coaches;

    #[ORM\OneToMany(mappedBy: 'user', targetEntity: MasterTeacher::class, orphanRemoval: true)]
    private Collection $masterTeachers;

    #[ORM\OneToMany(mappedBy: 'receiver', targetEntity: MailNotification::class, orphanRemoval: true)]
    private Collection $mailNotifications;

    #[ORM\OneToMany(mappedBy: 'sender', targetEntity: Message::class, orphanRemoval: true)]
    private Collection $sentMessages;

    #[ORM\OneToMany(mappedBy: 'receiver', targetEntity: MessageReceiver::class, orphanRemoval: true)]
    private Collection $messageReceivers;

    /**
     * @throws Exception
     */
    public function getDashboard(): string
    {
        if (in_array(User::ROLE_ADMIN, $this->getRoles())) {
            return AdminDashboardController::class;
        }
        if (in_array(User::ROLE_EDUCATOR, $this->getRoles())) {
            return EducatorDashboardController::class;
        }
        if (in_array(User::ROLE_STUDENT, $this->getRoles())) {
            return StudentDashboardController::class;
        }
        if (in_array(User::ROLE_GUARDIAN, $this->getRoles())) {
            return GuardianDashboardController::class;
        }
        throw new Exception('unknown role');
    }

    public function getStudent(): ?Student
    {
        if ($this->students->isEmpty()) {
            return null;
        }
        $students = $this->students->toArray();
        return reset($students);
    }
    public function getEducator(): ?Educator
    {
        if ($this->educators->isEmpty()) {
            return null;
        }
        $educator = $this->educators->toArray();
        return reset($educator);
    }
    public function getGuardian(): ?Guardian
    {
        if ($this->guardians->isEmpty()) {
            return null;
        }
        $guardian = $this->guardians->toArray();
        return reset($guardian);
    }
    public function getCoach(): ?Coach
    {
        if ($this->coaches->isEmpty()) {
            return null;
        }
        $coach = $this->coaches->toArray();
        return reset($coach);
    }
    public function getMasterTeacher(): ?MasterTeacher
    {
        if ($this->masterTeachers->isEmpty()) {
            return null;
        }
        $masterTeacher = $this->masterTeachers->toArray();
        return reset($masterTeacher);
    }
    public function __construct()
    {
        $this->students = new ArrayCollection();
        $this->educators = new ArrayCollection();
        $this->guardians = new ArrayCollection();
        $this->coaches = new ArrayCollection();
        $this->masterTeachers = new ArrayCollection();
        $this->mailNotifications = new ArrayCollection();
        $this->sentMessages = new ArrayCollection();
        $this->messageReceivers = new ArrayCollection();
    }

    public function __toString(): string
    {
        return $this->firstName . ' ' . $this->lastName. ' ' .$this->email;
    }

    public function getId(): ?int
    {
        return $this->id;
    }

    public function getEmail(): ?string
    {
        return $this->email;
    }

    public function setEmail(?string $email): static
    {
        $this->email = $email;

        return $this;
    }

    /**
     * A visual identifier that represents this user.
     *
     * @see UserInterface
     */
    public function getUserIdentifier(): string
    {
        return (string) $this->email;
    }

    /**
     * @see UserInterface
     */
    public function getRoles(): array
    {
        $roles = $this->roles;
        // guarantee every user at least has ROLE_USER
        $roles[] = 'ROLE_USER';

        return array_unique($roles);
    }

    public function setRoles(array $roles): static
    {
        $this->roles = $roles;

        return $this;
    }

    public function hasRole(string $role): bool
    {
        return in_array($role, $this->roles);
    }

    /**
     * @see PasswordAuthenticatedUserInterface
     */
    public function getPassword(): string
    {
        return $this->password;
    }

    public function setPassword(string $password): static
    {
        $this->password = $password;

        return $this;
    }

    /**
     * @see UserInterface
     */
    public function eraseCredentials(): void
    {
        // If you store any temporary, sensitive data on the user, clear it here
        // $this->plainPassword = null;
    }
    public function isActive(): ?bool
    {
        return $this->active;
    }

    public function setActive(bool $active): static
    {
        $this->active = $active;

        return $this;
    }

    public function getFirstName(): ?string
    {
        return $this->firstName;
    }

    public function setFirstName(?string $firstName): static
    {
        $this->firstName = $firstName;

        return $this;
    }

    public function getLastName(): ?string
    {
        return $this->lastName;
    }

    public function setLastName(?string $lastName): static
    {
        $this->lastName = $lastName;

        return $this;
    }

    public function getPhoneCountry(): ?string
    {
        return $this->phoneCountry;
    }

    public function setPhoneCountry(?string $phoneCountry): static
    {
        $this->phoneCountry = $phoneCountry;

        return $this;
    }

    public function getPhoneNumber(): ?string
    {
        return $this->phoneNumber;
    }

    public function setPhoneNumber(?string $phoneNumber): static
    {
        $this->phoneNumber = $phoneNumber;

        return $this;
    }
    /**
     * @return Collection<int, Student>
     */
    public function getStudents(): Collection
    {
        return $this->students;
    }

    public function addStudent(Student $student): static
    {
        if (!$this->students->contains($student)) {
            $this->students->add($student);
            $student->setUser($this);
        }

        return $this;
    }

    public function removeStudent(Student $student): static
    {
        if ($this->students->removeElement($student)) {
            // set the owning side to null (unless already changed)
            if ($student->getUser() === $this) {
                $student->setUser(null);
            }
        }

        return $this;
    }

    /**
     * @return Collection<int, Educator>
     */
    public function getEducators(): Collection
    {
        return $this->educators;
    }

    public function addEducator(Educator $educator): static
    {
        if (!$this->educators->contains($educator)) {
            $this->educators->add($educator);
            $educator->setUser($this);
        }

        return $this;
    }

    public function removeEducator(Educator $educator): static
    {
        if ($this->educators->removeElement($educator)) {
            // set the owning side to null (unless already changed)
            if ($educator->getUser() === $this) {
                $educator->setUser(null);
            }
        }

        return $this;
    }

    /**
     * @return Collection<int, Guardian>
     */
    public function getGuardians(): Collection
    {
        return $this->guardians;
    }

    public function addGuardian(Guardian $guardian): static
    {
        if (!$this->guardians->contains($guardian)) {
            $this->guardians->add($guardian);
            $guardian->setUser($this);
        }

        return $this;
    }

    public function removeGuardian(Guardian $guardian): static
    {
        if ($this->guardians->removeElement($guardian)) {
            // set the owning side to null (unless already changed)
            if ($guardian->getUser() === $this) {
                $guardian->setUser(null);
            }
        }

        return $this;
    }

    /**
     * @return Collection<int, Coach>
     */
    public function getCoaches(): Collection
    {
        return $this->coaches;
    }

    public function addCoach(Coach $coach): static
    {
        if (!$this->coaches->contains($coach)) {
            $this->coaches->add($coach);
            $coach->setUser($this);
        }

        return $this;
    }

    public function removeCoach(Coach $coach): static
    {
        if ($this->coaches->removeElement($coach)) {
            // set the owning side to null (unless already changed)
            if ($coach->getUser() === $this) {
                $coach->setUser(null);
            }
        }

        return $this;
    }

    /**
     * @return Collection<int, MasterTeacher>
     */
    public function getMasterTeachers(): Collection
    {
        return $this->masterTeachers;
    }

    public function addMasterTeacher(MasterTeacher $masterTeacher): static
    {
        if (!$this->masterTeachers->contains($masterTeacher)) {
            $this->masterTeachers->add($masterTeacher);
            $masterTeacher->setUser($this);
        }

        return $this;
    }

    public function removeMasterTeacher(MasterTeacher $masterTeacher): static
    {
        if ($this->masterTeachers->removeElement($masterTeacher)) {
            // set the owning side to null (unless already changed)
            if ($masterTeacher->getUser() === $this) {
                $masterTeacher->setUser(null);
            }
        }

        return $this;
    }

    /**
     * @return Collection<int, MailNotification>
     */
    public function getMailNotifications(): Collection
    {
        return $this->mailNotifications;
    }

    public function addMailNotification(MailNotification $mailNotification): static
    {
        if (!$this->mailNotifications->contains($mailNotification)) {
            $this->mailNotifications->add($mailNotification);
            $mailNotification->setReceiver($this);
        }

        return $this;
    }

    public function removeMailNotification(MailNotification $mailNotification): static
    {
        if ($this->mailNotifications->removeElement($mailNotification)) {
            // set the owning side to null (unless already changed)
            if ($mailNotification->getReceiver() === $this) {
                $mailNotification->setReceiver(null);
            }
        }

        return $this;
    }

    /**
     * @return Collection<int, Message>
     */
    public function getSentMessages(): Collection
    {
        return $this->sentMessages;
    }

    public function addSentMessage(Message $message): static
    {
        if (!$this->sentMessages->contains($message)) {
            $this->sentMessages->add($message);
            $message->setSender($this);
        }

        return $this;
    }

    public function removeSentMessage(Message $message): static
    {
        if ($this->sentMessages->removeElement($message)) {
            // set the owning side to null (unless already changed)
            if ($message->getSender() === $this) {
                $message->setSender(null);
            }
        }

        return $this;
    }

    /**
     * @return Collection<int, MessageReceiver>
     */
    public function getMessageReceivers(): Collection
    {
        return $this->messageReceivers;
    }

    public function addMessageReceiver(MessageReceiver $messageReceiver): static
    {
        if (!$this->messageReceivers->contains($messageReceiver)) {
            $this->messageReceivers->add($messageReceiver);
            $messageReceiver->setReceiver($this);
        }

        return $this;
    }

    public function removeMessageReceiver(MessageReceiver $messageReceiver): static
    {
        if ($this->messageReceivers->removeElement($messageReceiver)) {
            // set the owning side to null (unless already changed)
            if ($messageReceiver->getReceiver() === $this) {
                $messageReceiver->setReceiver(null);
            }
        }

        return $this;
    }
}
