<?php

namespace App\EventSubscriber;

use App\Controller\Admin\Crud\MeetingCrudController;
use App\Controller\Guardian\Crud\EducatorCrudGuardianController;
use App\Controller\Guardian\Crud\LeavingHomeCrudGuardianController;
use App\Controller\Guardian\Crud\WalkOutCrudGuardianController;
use App\Controller\Student\Crud\EducatorCrudStudentController;
use App\Controller\Student\Crud\LeavingHomeCrudStudentController;
use App\Controller\Student\Crud\MeetingCrudStudentController;
use App\Controller\Student\Crud\WalkOutCrudStudentController;
use App\Entity\Attendance;
use App\Entity\Coach;
use App\Entity\Educator;
use App\Entity\Guardian;
use App\Entity\MailNotification;
use App\Entity\Meeting;
use App\Entity\Room;
use App\Entity\School;
use App\Entity\Student;
use App\Entity\Activity;
use App\Entity\StudentNote;
use App\Entity\User;
use App\Repository\MeetingRepository;
use App\Repository\StudentRepository;
use App\Services\Mailer;
use App\Services\NotificationService;
use App\Services\PushNotifier;
use App\Services\UserSubscriptionManager;
use DateTimeImmutable;
use Doctrine\ORM\EntityManagerInterface;
use Doctrine\ORM\NonUniqueResultException;
use EasyCorp\Bundle\EasyAdminBundle\Config\Action;
use EasyCorp\Bundle\EasyAdminBundle\Context\AdminContext;
use EasyCorp\Bundle\EasyAdminBundle\Event\AfterEntityPersistedEvent;
use EasyCorp\Bundle\EasyAdminBundle\Event\BeforeCrudActionEvent;
use EasyCorp\Bundle\EasyAdminBundle\Event\BeforeEntityDeletedEvent;
use EasyCorp\Bundle\EasyAdminBundle\Router\AdminUrlGenerator;
use Exception;
use Symfony\Component\EventDispatcher\EventSubscriberInterface;
use EasyCorp\Bundle\EasyAdminBundle\Event\BeforeEntityPersistedEvent;
use EasyCorp\Bundle\EasyAdminBundle\Event\BeforeEntityUpdatedEvent;
use Symfony\Component\HttpKernel\Exception\AccessDeniedHttpException;
use Symfony\Contracts\Translation\TranslatorInterface;

class EasyAdminSubscriber implements EventSubscriberInterface
{
    private StudentRepository $studentRepository;
    private Mailer $mailer;
    private PushNotifier $pushNotifier;
    private UserSubscriptionManager $userSubscriptionManager;
    private AdminUrlGenerator $adminUrlGenerator;
    private EntityManagerInterface $entityManager;
    private MeetingRepository $meetingRepository;
    private TranslatorInterface $translator;

    public function __construct(StudentRepository $studentRepository, EntityManagerInterface $entityManager, Mailer $mailer, PushNotifier $pushNotifier,
                                UserSubscriptionManager $userSubscriptionManager, AdminUrlGenerator $adminUrlGenerator,
                                MeetingRepository $meetingRepository, TranslatorInterface $translator)
    {
        $this->studentRepository = $studentRepository;
        $this->mailer = $mailer;
        $this->pushNotifier = $pushNotifier;
        $this->userSubscriptionManager = $userSubscriptionManager;
        $this->adminUrlGenerator = $adminUrlGenerator;
        $this->entityManager = $entityManager;
        $this->meetingRepository = $meetingRepository;
        $this->translator = $translator;
    }

    public static function getSubscribedEvents(): array
    {
        return [
            BeforeEntityPersistedEvent::class => ['setParametersBeforePersist'],
            BeforeEntityUpdatedEvent::class => ['setParametersBeforeUpdate'],
            AfterEntityPersistedEvent::class => ['afterPersist'],
            BeforeEntityDeletedEvent::class => ['beforeEntityDeleted'],
            BeforeCrudActionEvent::class => ['beforeActionEvent'],
        ];
    }
    public function beforeEntityDeleted(BeforeEntityDeletedEvent $event): void
    {
        $entity = $event->getEntityInstance();
        if ($entity instanceof School){
            // remove also classrooms from that school
            foreach ($entity->getClassrooms() as $classroom){
                $this->entityManager->remove($classroom);
            }
        }
    }

    public function setParametersBeforePersist(BeforeEntityPersistedEvent $event): void
    {
        $entity = $event->getEntityInstance();
        $this->setParametersForActivityEntity($entity);
        $this->setParametersForStudentEntity($entity);
        $this->setParametersForRoomEntity($entity);

    }

    public function setParametersBeforeUpdate(BeforeEntityUpdatedEvent $event): void
    {
        $entity = $event->getEntityInstance();
        $this->setParametersForActivityEntity($entity);
        $this->setParametersForStudentEntity($entity);
        $this->setParametersForRoomEntity($entity);

    }

    /**
     * @throws Exception
     */
    public function afterPersist(AfterEntityPersistedEvent $event): void
    {
        $entity = $event->getEntityInstance();
        if ($entity instanceof StudentNote){
            $notificationService = new NotificationService($this->userSubscriptionManager, $this->adminUrlGenerator, $this->pushNotifier, $this->translator);
            $notificationService->createNotificationsForStudentNote($this->mailer, $entity);
            $this->entityManager->flush();
        }
        if ($entity instanceof Educator || $entity instanceof Student || $entity instanceof Guardian){
            // created entity, send mail about setting password
            $this->mailer->saveMailNotification($entity->getUser(), $this->translator->trans('email.new_user_subject'), new DateTimeImmutable(), MailNotification::NEW_USER_TYPE);
            $this->entityManager->flush();

        }
    }

    /**
     * @throws NonUniqueResultException
     */
    public function beforeActionEvent(BeforeCrudActionEvent $event): void
    {
        $context = $event->getAdminContext();
        if ($context->getCrud()->getEntityFqcn() === Meeting::class && $context->getCrud()->getCurrentAction() === Action::DETAIL) {
            // setting meeting entity joined with meetingAttendances
            $meetingId = (int) $context->getEntity()->getPrimaryKeyValue();
            $meeting = $this->meetingRepository->findMeetingById($meetingId);
            $context->getEntity()->setInstance($meeting);
        }
        /** @var User $user */
        $user = $context->getUser();
        $this->checkIfUserHaveAccessToThisPage($context, $user);
    }
    private function setParametersForStudentEntity($entity): void
    {
        if ($entity instanceof Student) {
            if ($entity->getUser()->isActive()) {
                //set parents to active
                foreach ($entity->getGuardians() as $guardian) {
                    $guardian->getUser()->setActive(true);
                }
            }
            else {
                //student set to inactive
                // remove dormitoryCard
                foreach ($entity->getDormitoryCards() as $dormitoryCard) {
                    $dormitoryCard->setStudent(null);
                }
                //set parents to inactive, if not have other child active
                foreach ($entity->getGuardians() as $guardian) {
                    $activeChildren = $this->getActiveChildren($guardian);
                    if (count($activeChildren) === 0) {
                        $guardian->getUser()->setActive(false);
                    }
                }
            }
        }
    }
    private function setParametersForActivityEntity($entity): void
    {
        if ($entity instanceof Activity) {
            $students = $this->studentRepository->findStudentsByActivity($entity);
            $coach = $entity->getCoach();
            /** @var Student $student */
            foreach ($students as $student) {
                if ($this->checkIfRemoveCoachFromStudent($student, $entity, $coach)){
                    $student->removeCoach($entity->getCoach());
                }
            }
            foreach ($entity->getStudents() as $student) {
                $student->addCoach($entity->getCoach());
            }
        }
    }
    private function setParametersForRoomEntity($entity): void
    {
        if ($entity instanceof Room) {
            $doorNumber = (string) $entity->getDoorNumber();
            $entity->setFloor($doorNumber[0]);
        }
    }
    private function checkIfRemoveCoachFromStudent(Student $student, Activity $entity, Coach $coach): bool
    {
        foreach ($student->getActivities() as $activity) {
            if ($activity !== $entity && $activity->getCoach() === $coach) {
                return false;
            }
        }
        return true;
    }

    private function getActiveChildren(Guardian $guardian): array
    {
        $activeChildren = [];
        foreach ($guardian->getChildren() as $child) {
            if ($child->getUser()->isActive()) {
                $activeChildren[] = $child;
            }
        }
        return $activeChildren;
    }

    private function checkIfUserHaveAccessToThisPage(AdminContext $context, User $user): void
    {
        // making sure that guardians and students only see detail page to which they are entitled
        // (Educators, Walkouts, Departures, Meetings (only students))
        if ($user->getStudent() === null && $user->getGuardian() === null){
            return;
        }
        if ($context->getCrud()->getCurrentAction() === Action::DETAIL) {
            $controllerFqcn = $context->getCrud()->getControllerFqcn();
            $entityInstance = $context->getEntity()->getInstance();
            switch ($context->getCrud()->getEntityFqcn()){
                case Meeting::class: {
                    if ($controllerFqcn === MeetingCrudStudentController::class) {
                        /** @var Meeting $meeting */
                        $meeting = $entityInstance;
                        if (!$meeting->isStudentInMessageReceivers($user->getStudent())) {
                            throw new AccessDeniedHttpException();
                        }
                    }
                    break;
                }
                case Educator::class: {
                    /** @var Educator $educator */
                    $educator = $entityInstance;
                    switch ($controllerFqcn){
                        case EducatorCrudStudentController::class:
                        case EducatorCrudGuardianController::class: {
                            if (!$educator->getUser()->isActive()) {
                                throw new AccessDeniedHttpException();
                            }
                            break;
                        }
                    }
                    break;
                }
                case Attendance::class: {
                    /** @var Attendance $attendance */
                    $attendance = $entityInstance;
                    switch ($controllerFqcn){
                        case LeavingHomeCrudGuardianController::class: {
                            if ($attendance->getAction() !== Attendance::DEPARTURE) {
                                throw new AccessDeniedHttpException();
                            }
                            if (!$user->getGuardian()->getChildren()->contains($attendance->getStudent())) {
                                throw new AccessDeniedHttpException();
                            }
                            break;
                        }
                        case LeavingHomeCrudStudentController::class: {
                            if ($attendance->getAction() !== Attendance::DEPARTURE) {
                                throw new AccessDeniedHttpException();
                            }
                            if ($attendance->getStudent() !== $user->getStudent()) {
                                throw new AccessDeniedHttpException();
                            }
                            break;
                        }
                        case WalkOutCrudGuardianController::class: {
                            if ($attendance->getAction() !== Attendance::WALK) {
                                throw new AccessDeniedHttpException();
                            }
                            if (!$user->getGuardian()->getChildren()->contains($attendance->getStudent())) {
                                throw new AccessDeniedHttpException();
                            }
                            break;
                        }
                        case WalkOutCrudStudentController::class: {
                            if ($attendance->getAction() !== Attendance::WALK) {
                                throw new AccessDeniedHttpException();
                            }
                            if ($attendance->getStudent() !== $user->getStudent()) {
                                throw new AccessDeniedHttpException();
                            }
                            break;
                        }
                    }
                    break;
                }

            }
        }
    }
}