<?php
namespace App\Form\Attendance;

use App\Form\AutocompleteField\EducatorAutocompleteField;
use App\Repository\RoomRepository;
use Symfony\Component\Form\AbstractType;
use Symfony\Component\Form\Event\PostSubmitEvent;
use Symfony\Component\Form\Extension\Core\Type\CheckboxType;
use Symfony\Component\Form\Extension\Core\Type\ChoiceType;
use Symfony\Component\Form\Extension\Core\Type\DateType;
use Symfony\Component\Form\Extension\Core\Type\SubmitType;
use Symfony\Component\Form\FormBuilderInterface;
use Symfony\Component\Form\FormError;
use Symfony\Component\Form\FormEvents;
use Symfony\Component\Validator\Constraints\NotBlank;
use Symfony\Contracts\Translation\TranslatableInterface;
use Symfony\Contracts\Translation\TranslatorInterface;

class ShowAttendanceType extends AbstractType
{
    private RoomRepository $roomRepository;
    private TranslatorInterface $translator;

    public function __construct(RoomRepository $roomRepository, TranslatorInterface $translator)
    {
        $this->roomRepository = $roomRepository;
        $this->translator = $translator;
    }

    /**
     * @throws \DateMalformedStringException
     */
    public function buildForm(FormBuilderInterface $builder, array $options): void
    {
        $date = $builder->getData()['date'];
        $app_starting_year = $builder->getData()['start'];
        $educator = $builder->getData()['educator'];

        $floors = $this->roomRepository->findDistinctFloors();
        $builder
            ->add('educator', EducatorAutocompleteField::class, [
                'data' => $educator,
            ])
            ->add('date', DateType::class, [
                'label' => 'entity.attendance.date',
                'widget' => 'single_text',
                'input' => 'datetime_immutable',
                'constraints' => [
                    new NotBlank(),
                ],
                'attr' => [
                    'min' => $app_starting_year. '-01-01',
                    'max' => $date->format('Y-m-d'),
                ]
            ])
            ->add('floor', ChoiceType::class, [
                'label' => 'entity.room.floor',
                'choices' => array_combine($floors, $floors),
                'placeholder' => 'entity.room.choose_floor',

            ])
            ->add('wholeMonth', CheckboxType::class, [
                'label' => 'entity.attendance.whole_month',
                'label_attr' => ['class' => 'checkbox-switch'],
            ])
            ->addEventListener(FormEvents::POST_SUBMIT, function (PostSubmitEvent $event): void {
                $educator = $event->getData()['educator'];
                $floor = $event->getData()['floor'];
                if (empty($educator) && empty($floor)) {
                    $event->getForm()->addError(new FormError($this->translator->trans('entity.attendance.form_choose_educator_or_floor')));
                }
                elseif (!empty($educator) && !empty($floor)) {
                    $floors = $this->roomRepository->findDistinctFloorsForEducator($educator);
                    if (!in_array($floor, $floors)) {
                        $event->getForm()->addError(new FormError($this->translator->trans('entity.attendance.form_empty_for_educator_and_floor')));
                    }
                }

            })
            ->add('save', SubmitType::class, [
                'label' => 'general.submit',
            ])
        ;
    }
}