<?php
namespace App\Form\Attendance;

use App\Entity\Educator;
use App\Entity\Student;
use App\Entity\User;
use Doctrine\ORM\EntityNotFoundException;
use Doctrine\ORM\EntityRepository;
use Symfony\Bridge\Doctrine\Form\Type\EntityType;
use Symfony\Bundle\SecurityBundle\Security;
use Symfony\Component\Form\AbstractType;
use Symfony\Component\Form\Extension\Core\Type\ChoiceType;
use Symfony\Component\Form\Extension\Core\Type\SubmitType;
use Symfony\Component\Form\FormBuilderInterface;
use Symfony\Component\Validator\Constraints\NotBlank;

class ShowStudentAttendanceGuardianType extends AbstractType
{
    private Security $security;

    public function __construct(Security $security)
    {

        $this->security = $security;
    }

    /**
     * @throws EntityNotFoundException
     */
    public function buildForm(FormBuilderInterface $builder, array $options): void
    {
        $date = $builder->getData()['date'];
        $month = $date->format('m');
        $year = $date->format('Y');
        $app_starting_year = $builder->getData()['start'];

        /** @var User $user */
        $user = $this->security->getUser();
        if ($user === null || $user->getGuardian() === null) {
            throw new EntityNotFoundException();
        }
        $builder->add('child', EntityType::class, [
            'label' => 'entity.guardian.child.singular_name',
            'class' => Student::class,
            'query_builder' => function (EntityRepository $er) use ($user) {
                return $er->createQueryBuilder('s')
                    ->addSelect('user_query')
                    ->leftJoin('s.user', 'user_query')
                    ->leftJoin('s.guardians', 'guardian')
                    ->andWhere('guardian.id = :guardian')
                    ->andWhere('user_query.active = :active')
                    ->setParameter('active', true)
                    ->setParameter('guardian', $user->getGuardian()->getId());
            },
            'placeholder' => 'entity.guardian.child.choose_one',
            'constraints' => [
                new NotBlank(),
            ],
        ]);

        $builder->add('month', ChoiceType::class, [
            'choices' => $this->getMonthChoices(),
            'placeholder' => 'general.choose_month',
            'label' => 'general.month',
            'data' => $month,
            'constraints' => [
                new NotBlank(),
            ],
        ]);

        $builder->add('year', ChoiceType::class, [
            'choices' => $this->getYearChoices($app_starting_year),
            'data' => $year,
            'placeholder' => 'general.choose_year',
            'label' => 'general.year',
            'constraints' => [
                new NotBlank(),
            ],
        ]);

        $builder
            ->add('save', SubmitType::class, [
                'label' => 'general.filter',
            ])
        ;
    }
    private function getMonthChoices(): array
    {
        return [
            'general.january'   => '01',
            'general.february'  => '02',
            'general.march'     => '03',
            'general.april'     => '04',
            'general.may'       => '05',
            'general.june'      => '06',
            'general.july'      => '07',
            'general.august'    => '08',
            'general.september' => '09',
            'general.october'   => '10',
            'general.november'  => '11',
            'general.december'  => '12',
        ];
    }

    private function getYearChoices(int $app_starting_year): array
    {
        $currentYear = date('Y');
        $years = range($app_starting_year, $currentYear);

        return array_combine($years, $years);
    }
}