<?php
namespace App\Form;

use App\Entity\Educator;
use App\Entity\Message;
use App\Entity\User;
use App\Repository\UserRepository;
use Symfony\Bridge\Doctrine\Form\Type\EntityType;
use Symfony\Bundle\SecurityBundle\Security;
use Symfony\Component\Form\AbstractType;
use Symfony\Component\Form\Extension\Core\Type\ChoiceType;
use Symfony\Component\Form\Extension\Core\Type\SubmitType;
use Symfony\Component\Form\Extension\Core\Type\TextareaType;
use Symfony\Component\Form\FormBuilderInterface;
use Symfony\Component\Form\FormEvent;
use Symfony\Component\Form\FormEvents;
use Symfony\Component\Form\FormInterface;
use Symfony\Component\Validator\Constraints\NotBlank;

class CreateMessageType extends AbstractType
{
    private Security $security;
    private UserRepository $userRepository;

    public function __construct(Security $security, UserRepository $userRepository)
    {

        $this->security = $security;
        $this->userRepository = $userRepository;
    }
    public function buildForm(FormBuilderInterface $builder, array $options): void
    {
        $builder->add('message', TextareaType::class, [
            'constraints' => [
                new NotBlank(),
            ],
            'label' => 'entity.message.text'
        ]);
        $receiverChoices = [
            'entity.message.all_dormitory' => Message::ALL_TYPE,
            'entity.educator.plural_name' => Message::EDUCATOR_TYPE,
            'entity.student.plural_name' => Message::STUDENT_TYPE,
            'entity.guardian.plural_name' => Message::GUARDIAN_TYPE,
            'entity.message.students_and_guardians' => Message::STUDENT_AND_GUARDIAN_TYPE,
        ];
        /** @var User $user */
        $user = $this->security->getUser();
        if (in_array(User::ROLE_GUARDIAN, $user->getRoles()) || in_array(User::ROLE_STUDENT, $user->getRoles())) {
            $receiverChoices = [
                'entity.educator.plural_name' => Message::EDUCATOR_TYPE,
            ];
        }
        if (in_array(User::ROLE_EDUCATOR, $user->getRoles())) {
            $receiverChoices['entity.message.educator\'s_students'] = Message::EDUCATOR_STUDENTS_TYPE;
        }
        $builder->add('receiverType', ChoiceType::class, [
            'label' => 'entity.message.receiver_type',
            'choices' => $receiverChoices,
            'expanded' => true,
            'constraints' => [
                new NotBlank(),
            ],
        ]);
        $formModifier = function (FormInterface $form, string $receiverType = null, User $user = null, Educator $educator = null): void {
            $query_builder = null === $receiverType? null : $this->userRepository->getQueryBuilderForReceiverType($receiverType, $user, $educator);
            $form->add('receivers', EntityType::class, [
                'label' => 'entity.message.receivers',
                'class' => User::class,
                'query_builder' => function () use ($query_builder) {
                    return $query_builder;
                },
                'multiple' => true,
                'autocomplete' => true,
                'row_attr' => [
                    'class' => 'mb-3' . (null === $receiverType? ' d-none' : ''),
                ],
            ]);

        };

        $builder->addEventListener(
            FormEvents::PRE_SET_DATA,
            function (FormEvent $event) use ($formModifier): void {
                $formModifier($event->getForm());
            }
        );

        $builder->get('receiverType')->addEventListener(
            FormEvents::POST_SUBMIT,
            function (FormEvent $event) use ($formModifier, $user): void {
                $receiverType = $event->getForm()->getData(); // value of receiverType field
                $formModifier($event->getForm()->getParent(), $receiverType, $user, $user->getEducator());
            }
        );

        $builder
            ->add('save', SubmitType::class, [
                'label' => 'entity.message.button_send'
            ])
        ;
    }
}