<?php

namespace App\Form\DataTransformer;

use App\Exception\EmptyColumnsInImportMeetingsException;
use App\Services\Import\Import;
use DateTimeImmutable;
use Exception;
use Symfony\Component\Form\DataTransformerInterface;
use Symfony\Component\Form\Exception\TransformationFailedException;
use Symfony\Component\HttpFoundation\File\UploadedFile;
use Symfony\Component\Validator\Constraints as Assert;
use Symfony\Component\Validator\Validator\ValidatorInterface;


class FileToIndividualMeetingDataTransformer implements DataTransformerInterface
{
    const DATE_KEY = 'Datum';
    const ROOM_KEY = 'Miestnost';
    const EDUCATOR_KEY = 'Vychovavatel';
    const STUDENT_KEY = 'Ziak';
    public function __construct(
        private readonly Import             $import,
        private readonly ValidatorInterface $validator,
    ) {
    }

    /**
     * Transforms an object (array) to a file.
     *
     * @param  $value
     * @return string
     */
    public function transform($value): string
    {
        return "";
    }

    /**
     * Transforms a string (number) to an object (issue).
     *
     * @param  UploadedFile|null $value
     * @throws TransformationFailedException if object (issue) is not found.
     */
    public function reverseTransform($value): ?array
    {
        if ($value === null) {
            return null;
        }
        $sequence = range('A', 'D');
        try {
            $importedData = $this->import->upload($value, self::getKeys(), $sequence);
            $this->validateMeeting($importedData);

        } catch (TransformationFailedException $e){
            throw $e;
        }
        catch (Exception $e){
            $failure = new TransformationFailedException($e->getMessage());
            $failure->setInvalidMessage($e->getMessage());
            throw $failure;
        }
        return $importedData;
    }
    public static function getKeys(): array
    {
        return[self::DATE_KEY, self::ROOM_KEY, self::STUDENT_KEY, self::EDUCATOR_KEY];
    }

    /**
     * @throws EmptyColumnsInImportMeetingsException
     * @throws Exception
     */
    public function validateMeeting(array $importedIndividualMeetings): void
    {
        $invalidEmails = [];
        $currentDate = new DateTimeImmutable();
        foreach ($importedIndividualMeetings as $importedIndividualMeeting) {
            if (empty($importedIndividualMeeting[self::DATE_KEY]) || empty($importedIndividualMeeting[self::ROOM_KEY]) ||
                empty($importedIndividualMeeting[self::STUDENT_KEY]) || empty($importedIndividualMeeting[self::EDUCATOR_KEY])) {
                throw new EmptyColumnsInImportMeetingsException();
            }
            foreach ([$importedIndividualMeeting[self::STUDENT_KEY], $importedIndividualMeeting[self::EDUCATOR_KEY]] as $email) {
                $errors = $this->validator->validate($email, new Assert\Email());
                if ($errors->count() > 0) {
                    $invalidEmails[] = $email;
                }
            }
            $errors = $this->validator->validate($importedIndividualMeeting[self::DATE_KEY], new Assert\DateTime('d.m.Y H:i'));
            if ($errors->count() > 0) {
                throw new TransformationFailedException('', 0, null, 'data_transformer.invalid_date', [
                    '%date%' => $importedIndividualMeeting[self::DATE_KEY],
                ]);
            }

            $meetingDate = new DateTimeImmutable($importedIndividualMeeting[self::DATE_KEY]);
            if ($meetingDate->format('Y-m-d') < $currentDate->format('Y-m-d')){
                throw new TransformationFailedException('', 0, null, 'data_transformer.not_future_date', [
                    '%date%' => $importedIndividualMeeting[self::DATE_KEY],
                ]);
            }
        }
        if (!empty($invalidEmails)) {
            throw new TransformationFailedException('', 0, null, 'data_transformer.invalid_email_address', [
                '%email%' => implode(', ', $invalidEmails),
            ]);
        }
    }
}