<?php

namespace App\Form\DataTransformer;
use App\Entity\Schedule;
use App\Services\Import\Import;
use Exception;
use Symfony\Component\Form\DataTransformerInterface;
use Symfony\Component\Form\Exception\TransformationFailedException;
use Symfony\Component\HttpFoundation\File\UploadedFile;

class FileToScheduleDataTransformer implements DataTransformerInterface
{
    const MONDAY_KEY = 'PO';
    const TUESDAY_KEY = 'UT';
    const WEDNESDAY_KEY = 'ST';
    const THURSDAY_KEY = 'ŠT';
    const FRIDAY_KEY = 'PI';
    const SATURDAY_KEY = 'SO';
    const SUNDAY_KEY = 'NE';
    public function __construct(
        private readonly Import             $import,
        private readonly int                $app_week_cycle,
    ) {
    }

    /**
     * Transforms an object (array) to a file.
     *
     * @param  $value
     * @return string
     */
    public function transform($value): string
    {
        return "";
    }

    /**
     * Transforms a string (number) to an object (issue).
     *
     * @param  UploadedFile|null $value
     * @throws TransformationFailedException if object (issue) is not found.
     */
    public function reverseTransform($value): ?array
    {
        if ($value === null) {
            return null;
        }
        $sequence = range('A', 'G');
        try {
            $importedData = $this->import->upload($value, self::getKeys(), $sequence);
            $this->validateSchedule($importedData);

        } catch (TransformationFailedException $e){
            throw $e;
        }
        catch (Exception $e){
            $failure = new TransformationFailedException($e->getMessage());
            $failure->setInvalidMessage($e->getMessage());
            throw $failure;
        }
        return $importedData;
    }
    public static function getKeys(): array
    {
        return [self::MONDAY_KEY, self::TUESDAY_KEY, self::WEDNESDAY_KEY, self::THURSDAY_KEY, self::FRIDAY_KEY, self::SATURDAY_KEY, self::SUNDAY_KEY];
    }
    /**
     * @throws Exception
     */
    private function validateSchedule(array $importedSchedules): void
    {
        if (count($importedSchedules) != $this->app_week_cycle) {
            throw new TransformationFailedException('', 0, null, 'data_transformer.wrong_number_of_rows', [
                '%number%' => $this->app_week_cycle,
            ]);
        }
        $allowedValues = [Schedule::P7_KEY, Schedule::P8_KEY, Schedule::N8_KEY, Schedule::N10_KEY, Schedule::R8_KEY, Schedule::P_KEY];
        foreach ($importedSchedules as $importedSchedule) {
            foreach ($importedSchedule as $day) {
                if (empty($day)) {
                    continue;
                }
                if (!in_array($day, $allowedValues)) {
                    throw new TransformationFailedException('', 0, null, 'data_transformer.invalid_values', [
                        '%key%' => $day,
                        '%values%' => implode(', ', $allowedValues),
                    ]);
                }
            }
        }
    }
}