<?php

namespace App\Form\DataTransformer;
use App\Services\Import\Import;
use Exception;
use Symfony\Component\Form\DataTransformerInterface;
use Symfony\Component\Form\Exception\TransformationFailedException;
use Symfony\Component\HttpFoundation\File\UploadedFile;
use Symfony\Component\Validator\Constraints as Assert;
use Symfony\Component\Validator\Validator\ValidatorInterface;

class FileToTrainingDataTransformer implements DataTransformerInterface
{
    const STUDENT_EMAIL_KEY = 'Studenti';
    const COACH_EMAIL_KEY = 'Trener email';
    const FROM_KEY = 'Od';
    const TO_KEY = 'Do';
    const DAY_KEY = 'Den';
    const DAY_VALUES = ['PO', 'UT', 'ST', 'ŠT', 'PI', 'SO', 'NE'];

    public function __construct(
        private readonly Import             $import,
        private readonly ValidatorInterface $validator,
    ) {
    }

    /**
     * Transforms an object (array) to a file.
     *
     * @param  $value
     * @return string
     */
    public function transform($value): string
    {
        return "";
    }

    /**
     * Transforms a string (number) to an object (issue).
     *
     * @param  UploadedFile|null $value
     * @throws TransformationFailedException if object (issue) is not found.
     */
    public function reverseTransform($value): ?array
    {
        if ($value === null) {
            return null;
        }
        $sequence = range('A', 'E');
        try {
            $importedData = $this->import->upload($value, self::getKeys(), $sequence);
            $this->validateTraining($importedData);

        } catch (TransformationFailedException $e){
            throw $e;
        }
        catch (Exception $e){
            $failure = new TransformationFailedException($e->getMessage());
            $failure->setInvalidMessage($e->getMessage());
            throw $failure;
        }

        return $importedData;
    }
    public static function getKeys(): array
    {
        return [self::COACH_EMAIL_KEY, self::FROM_KEY, self::TO_KEY, self::DAY_KEY, self::STUDENT_EMAIL_KEY];
    }
    /**
     * @throws Exception
     */
    private function validateTraining(array $importedTrainings): void
    {
        $pattern = '/^([0-9]|0[0-9]|1[0-9]|2[0-3]):[0-5][0-9]$/';
        $invalidEmails = [];

        foreach ($importedTrainings as $importedTraining) {
            if (empty($importedTraining[self::COACH_EMAIL_KEY])) {
                throw new Exception('data_transformer.empty_trainer_email');
            }
            if (empty($importedTraining[self::STUDENT_EMAIL_KEY])) {
                throw new Exception('data_transformer.empty_student_email');
            }
            if (empty($importedTraining[self::FROM_KEY]) || empty($importedTraining[self::TO_KEY]) || empty($importedTraining[self::DAY_KEY])) {
                throw new Exception('data_transformer.empty_from_to_day');
            }
            $students = explode(',', $importedTraining[self::STUDENT_EMAIL_KEY]);
            foreach (array_merge($students, [$importedTraining[self::COACH_EMAIL_KEY]])as $email) {
                $errors = $this->validator->validate($email, new Assert\Email());
                if ($errors->count() > 0) {
                    $invalidEmails[] = $email;
                }
            }
            foreach ([$importedTraining[self::TO_KEY], $importedTraining[self::FROM_KEY]] as $value) {
                $violations = $this->validator->validate($value, new Assert\Regex($pattern));
                if ($violations->count() > 0) {
                    throw new Exception('data_transformer.format_from_to');

                }
            }
            if (!in_array($importedTraining[self::DAY_KEY], self::DAY_VALUES)) {
                throw new TransformationFailedException('', 0, null, 'data_transformer.invalid_values', [
                    '%key%' => $importedTraining[self::DAY_KEY],
                    '%values%' => implode(', ', self::DAY_VALUES),
                ]);
            }
        }
        if (!empty($invalidEmails)) {
            throw new TransformationFailedException('', 0, null, 'data_transformer.invalid_email_address', [
                '%email%' => implode(', ', $invalidEmails),
            ]);
        }

    }
}