<?php

namespace App\Repository;

use App\Entity\Attendance;
use App\Entity\Educator;
use App\Entity\Student;
use DateTimeImmutable;
use Doctrine\Bundle\DoctrineBundle\Repository\ServiceEntityRepository;
use Doctrine\ORM\NonUniqueResultException;
use Doctrine\Persistence\ManagerRegistry;

/**
 * @extends ServiceEntityRepository<Attendance>
 *
 * @method Attendance|null find($id, $lockMode = null, $lockVersion = null)
 * @method Attendance|null findOneBy(array $criteria, array $orderBy = null)
 * @method Attendance[]    findAll()
 * @method Attendance[]    findBy(array $criteria, array $orderBy = null, $limit = null, $offset = null)
 */
class AttendanceRepository extends ServiceEntityRepository
{
    public function __construct(ManagerRegistry $registry)
    {
        parent::__construct($registry, Attendance::class);
    }
    public function findByStudentAndDate(Student $student, DateTimeImmutable $date): array
    {
        return $this->createQueryBuilder('a')
            ->andWhere('a.student = :student')
            ->setParameter('student', $student->getId())
            ->andWhere('(DATE(a.time) = :date OR (DATE(a.time) <= :date AND DATE(a.validUntil) >= :date))')
            ->setParameter('date', $date->format('Y-m-d'))
            ->andWhere('NOT(a.action = :present OR a.action = :absent OR a.action = :check)')
            ->setParameter('present', Attendance::PRESENT)
            ->setParameter('absent', Attendance::ABSENT)
            ->setParameter('check', Attendance::EVENING_CHECK)
            ->addOrderBy('a.time', 'ASC')
            ->getQuery()
            ->getResult();
    }
    public function findByStudentsArrayAndDate(array $students, DateTimeImmutable $date)
    {
        return $this->createQueryBuilder('a')
            ->andWhere('a.student IN (:students)')
            ->setParameter('students', $students)
            ->andWhere('(DATE(a.time) = :date OR (DATE(a.time) <= :date AND DATE(a.validUntil) >= :date))')
            ->setParameter('date', $date->format('Y-m-d'))
            ->andWhere('NOT(a.action = :present OR a.action = :absent OR a.action = :check)')
            ->setParameter('present', Attendance::PRESENT)
            ->setParameter('absent', Attendance::ABSENT)
            ->setParameter('check', Attendance::EVENING_CHECK)
            ->addOrderBy('a.time', 'ASC')
            ->getQuery()
            ->getResult();
    }

    public function findLastPresentOrAbsentFromArray(array $students, DateTimeImmutable $date)
    {
        $subQuery = $this->createQueryBuilder('lead')
            ->select('MAX(lead.time)')
            ->andWhere('lead.student IN (:students)')
            ->andWhere('DATE(lead.time) <= :date')
            ->andWhere('(lead.action = :present OR lead.action = :absent)')
            ->andWhere('a.student = lead.student');
        ;

        $query = $this->createQueryBuilder('a');
        $query
            ->where($query->expr()->eq('a.time', '(' . $subQuery->getDQL() . ')'))
            ->andWhere('(a.action = :present OR a.action = :absent)')
            ->setParameter('students', $students)
            ->setParameter('date', $date->format('Y-m-d'))
            ->setParameter('present', Attendance::PRESENT)
            ->setParameter('absent', Attendance::ABSENT)
        ;

        return $query->getQuery()->getResult();
    }

    /**
     * @throws NonUniqueResultException
     */
    public function findLastPresentOrAbsent(Student $student, DateTimeImmutable $date)
    {
        return $this->createQueryBuilder('a')
            ->andWhere('a.student = :student')
            ->setParameter('student', $student)
            ->andWhere('DATE(a.time) <= :date')
            ->setParameter('date', $date->format('Y-m-d'))
            ->andWhere('( a.action = :present OR a.action = :absent )')
            ->setParameter('present', Attendance::PRESENT)
            ->setParameter('absent', Attendance::ABSENT)
            ->addOrderBy('a.time', 'DESC')
            ->setMaxResults(1)
            ->getQuery()
            ->getOneOrNullResult();
    }
    public function findEveningCheck(array $students, DateTimeImmutable $date)
    {
        $subQuery = $this->createQueryBuilder('lead')
            ->select('MAX(lead.time)')
            ->andWhere('lead.student IN (:students)')
            ->andWhere('DATE(lead.time) = :date')
            ->andWhere('lead.action = :check')
            ->andWhere('a.student = lead.student');

        $query = $this->createQueryBuilder('a');
        $query
            ->where($query->expr()->eq('a.time', '(' . $subQuery->getDQL() . ')'))
            ->andWhere('a.action = :check')
            ->setParameter('students', $students)
            ->setParameter('date', $date->format('Y-m-d'))
            ->setParameter('check', Attendance::EVENING_CHECK)

        ;
        return $query->getQuery()->getResult();
    }

    /**
     * @throws NonUniqueResultException
     */
    public function findEveningCheckForStudent(Student $student, DateTimeImmutable $date)
    {
        return $this->createQueryBuilder('a')
            ->andWhere('a.student = :student')
            ->setParameter('student', $student)
            ->andWhere('DATE(a.time) = :date')
            ->setParameter('date', $date->format('Y-m-d'))
            ->andWhere('a.action = :check')
            ->setParameter('check', Attendance::EVENING_CHECK)
            ->addOrderBy('a.time', 'DESC')
            ->setMaxResults(1)
            ->getQuery()
            ->getOneOrNullResult();
    }

    /**
     * @throws NonUniqueResultException
     */
    public function findAbsentOrPresentForTime(Student $student, DateTimeImmutable $date)
    {
        return $this->createQueryBuilder('a')
            ->andWhere('a.student = :student')
            ->setParameter('student', $student)
            ->andWhere('a.time <= :date')
            ->setParameter('date', $date->format('Y-m-d H:i:s'))
            ->andWhere('(a.action = :present OR a.action = :absent )')
            ->setParameter('present', Attendance::PRESENT)
            ->setParameter('absent', Attendance::ABSENT)
            ->addOrderBy('a.time', 'DESC')
            ->setMaxResults(1)
            ->getQuery()
            ->getOneOrNullResult();
    }

    /**
     * @throws NonUniqueResultException
     */
    public function findPresentActionToFuture(Student $student, DateTimeImmutable $date)
    {
        return $this->createQueryBuilder('a')
            ->andWhere('a.student = :student')
            ->setParameter('student', $student)
            ->andWhere('a.time >= :date')
            ->setParameter('date', $date->format('Y-m-d H:i:s'))
            ->andWhere('a.action = :present')
            ->setParameter('present', Attendance::PRESENT)
            ->addOrderBy('a.time', 'DESC')
            ->setMaxResults(1)
            ->getQuery()
            ->getOneOrNullResult();
    }
}
