<?php

namespace App\Repository;

use App\Entity\Message;
use App\Entity\MessageReceiver;
use App\Entity\User;
use Doctrine\Bundle\DoctrineBundle\Repository\ServiceEntityRepository;
use Doctrine\ORM\AbstractQuery;
use Doctrine\ORM\Query;
use Doctrine\ORM\Query\ResultSetMappingBuilder;
use Doctrine\ORM\QueryBuilder;
use Doctrine\Persistence\ManagerRegistry;

/**
 * @extends ServiceEntityRepository<Message>
 *
 * @method Message|null find($id, $lockMode = null, $lockVersion = null)
 * @method Message|null findOneBy(array $criteria, array $orderBy = null)
 * @method Message[]    findAll()
 * @method Message[]    findBy(array $criteria, array $orderBy = null, $limit = null, $offset = null)
 */
class MessageRepository extends ServiceEntityRepository
{
    public function __construct(ManagerRegistry $registry)
    {
        parent::__construct($registry, Message::class);
    }
    public function getWithSearchQueryBuilderSentMessages(?string $term, User $user): QueryBuilder
    {
        $qb = $this->createQueryBuilder('m')
            ->leftJoin('m.sender', 'sender')
            ->leftJoin('m.messageReceivers', 'receivers')
            ->leftJoin('m.messageReceivers', 'receivers_query')
            ->leftJoin('receivers_query.receiver', 'receiver_user')
            ->addSelect('sender', 'receivers')
            ->andWhere('sender.id = :user')
            ->setParameter('user', $user->getId());
        if ($term) {
            $qb->andWhere('(receiver_user.firstName LIKE :term OR receiver_user.lastName LIKE :term OR receiver_user.email LIKE :term)')
                ->setParameter('term', '%' . $term . '%')
            ;
        }
        return $qb
            ->orderBy('m.date', 'DESC')
            ;
    }

    public function getWithSearchQueryBuilderReceivedMessages(?string $term, User $user): QueryBuilder
    {
        $qb = $this->createQueryBuilder('m')
            ->leftJoin('m.sender', 'sender')
            ->leftJoin('m.messageReceivers', 'receivers')
            ->leftJoin('m.messageReceivers', 'receivers_query')
            ->addSelect('sender', 'receivers')
            ->andWhere('receivers_query.receiver = :user')
            ->setParameter('user', $user->getId());
        if ($term) {
            $qb->andWhere('(sender.firstName LIKE :term OR sender.lastName LIKE :term OR sender.email LIKE :term)')
                ->setParameter('term', '%' . $term . '%')
            ;
        }
        return $qb
            ->orderBy('m.date', 'DESC')
            ;
    }

    public function findUnreadMessagesForUser(User $user): array|float|int|string
    {
        return $this->createQueryBuilder('m')
            ->select('m.id')
            ->leftJoin('m.messageReceivers', 'receivers')
            ->andWhere('receivers.receiver = :user')
            ->andWhere('receivers.isRead = :is_read')
            ->setParameter('user', $user->getId())
            ->setParameter('is_read', false)
            ->getQuery()
            ->getSingleColumnResult();
    }

    public function getMessagesByIds(array $messagesIds, User $user)
    {
        return $this->createQueryBuilder('m')
            ->addSelect('receivers', 'user')
            ->leftJoin('m.messageReceivers', 'receivers_condition')
            ->leftJoin('m.messageReceivers', 'receivers')
            ->leftJoin('receivers.receiver', 'user')
            ->andWhere('m.id IN (:messagesIds)')
            ->andWhere('(:user IN (receivers_condition.receiver) OR :user = m.sender)')
            ->setParameter('messagesIds', $messagesIds)
            ->setParameter('user', $user->getId())
            ->getQuery()
            ->getResult();
    }

    public function getMessagesIdByRoot(Message $root): array|float|int|string
    {
        $nativeQuery =
            'WITH RECURSIVE message_hierarchy AS (
              SELECT id
              FROM message
              WHERE root_id = :root AND parent_id IS NULL
              
              UNION ALL

              SELECT m.id
              FROM message m
              INNER JOIN message_hierarchy mh ON m.parent_id = mh.id
            )
            SELECT id FROM message_hierarchy AS mh;';

        $rsm = new ResultSetMappingBuilder($this->_em);
        $rsm->addEntityResult(Message::class, 'u');
        $rsm->addFieldResult('u', 'id', 'id');

        $query = $this->_em->createNativeQuery($nativeQuery, $rsm);
        $query->setParameter('root', $root->getId());

        return $query->getSingleColumnResult();
    }
}
