<?php

namespace App\Repository;

use App\Entity\Educator;
use App\Entity\Guardian;
use App\Entity\Student;
use App\Entity\Activity;
use Doctrine\Bundle\DoctrineBundle\Repository\ServiceEntityRepository;
use Doctrine\ORM\NonUniqueResultException;
use Doctrine\Persistence\ManagerRegistry;

/**
 * @extends ServiceEntityRepository<Student>
 *
 * @method Student|null find($id, $lockMode = null, $lockVersion = null)
 * @method Student|null findOneBy(array $criteria, array $orderBy = null)
 * @method Student[]    findAll()
 * @method Student[]    findBy(array $criteria, array $orderBy = null, $limit = null, $offset = null)
 */
class StudentRepository extends ServiceEntityRepository
{
    public function __construct(ManagerRegistry $registry)
    {
        parent::__construct($registry, Student::class);
    }

    /**
     * @return Student[] Returns an array of Student objects
     */
    public function findByEducator($educator): array
    {
        return $this->createQueryBuilder('s')
            ->andWhere('s.educator = :value')
            ->setParameter('value', $educator)
            ->orderBy('s.id', 'ASC')
            ->getQuery()
            ->getResult()
        ;
    }
    public function getStudentsByFloorOrEducatorOrBoth(?int $floor, ?Educator $educator): array
    {
        $queryBuilder = $this->createQueryBuilder('s')
            ->select('s, u, r, a')
            ->leftJoin('s.room', 'r')
            ->leftJoin('s.user', 'u')
            ->leftJoin('s.activities', 'a');

        if ($floor !== null) {
            $queryBuilder
                ->andWhere('r.floor = :floorId')
                ->setParameter('floorId', $floor);
        }

        if ($educator !== null) {
            $queryBuilder
                ->andWhere('s.educator = :educatorId')
                ->setParameter('educatorId', $educator->getId());
        }
        $queryBuilder
            ->andWhere('s.room IS NOT NULL')
            ->addOrderBy('r.doorNumber', 'ASC')
            ->addOrderBy('u.lastName', 'ASC');

        return $queryBuilder->getQuery()->getResult();
    }
    public function findStudentsByActivity(Activity $activity)
    {
        return $this->createQueryBuilder('s')
            ->leftJoin('s.activities', 't')
            ->andWhere('t.id = :activity')
            ->setParameter('activity', $activity->getId())
            ->getQuery()
            ->getResult();
    }

    public function findActiveStudentsWithRoom(): array
    {
        return $this->createQueryBuilder('s')
            ->leftJoin('s.user', 'u')
            ->andWhere('u.active = :active')
            ->andWhere('s.room IS NOT NULL')
            ->setParameter('active', true)
            ->getQuery()
            ->getResult();
    }

    /**
     * @throws NonUniqueResultException
     */
    public function findActiveStudentWithRoomAndDormitoryCard(int $card_id)
    {
        return $this->createQueryBuilder('s')
            ->leftJoin('s.user', 'u')
            ->leftJoin('s.dormitoryCards', 'dormitory_card')
            ->andWhere('u.active = :active')
            ->andWhere('dormitory_card.number = :card')
            ->andWhere('s.room IS NOT NULL')
            ->setParameter('active', true)
            ->setParameter('card', $card_id)
            ->getQuery()
            ->getOneOrNullResult();
    }
    public function findAllJoinedActiveUser() {
        return $this->createQueryBuilder('s')
            ->select('s', 'u')
            ->leftJoin('s.user', 'u')
            ->andWhere('u.active = :active')
            ->setParameter('active', true)
            ->getQuery()
            ->getResult();
    }

    public function findChildrenForGuardian(Guardian $guardian)
    {
        return $this->createQueryBuilder('s')
            ->select('s', 'u')
            ->leftJoin('s.user', 'u')
            ->andWhere('u.active = :active')
            ->setParameter('active', true)
            ->leftJoin('s.guardians', 'guardian')
            ->andWhere('guardian.id = :guardian')
            ->setParameter('guardian', $guardian->getId())
            ->getQuery()
            ->getResult();
    }
}
