<?php

namespace App\Repository;

use App\Entity\Educator;
use App\Entity\Message;
use App\Entity\User;
use Doctrine\Bundle\DoctrineBundle\Repository\ServiceEntityRepository;
use Doctrine\ORM\NonUniqueResultException;
use Doctrine\ORM\QueryBuilder;
use Doctrine\Persistence\ManagerRegistry;
use Symfony\Component\Security\Core\Exception\UnsupportedUserException;
use Symfony\Component\Security\Core\User\PasswordAuthenticatedUserInterface;
use Symfony\Component\Security\Core\User\PasswordUpgraderInterface;

/**
 * @extends ServiceEntityRepository<User>
* @implements PasswordUpgraderInterface<User>
 *
 * @method User|null find($id, $lockMode = null, $lockVersion = null)
 * @method User|null findOneBy(array $criteria, array $orderBy = null)
 * @method User[]    findAll()
 * @method User[]    findBy(array $criteria, array $orderBy = null, $limit = null, $offset = null)
 */
class UserRepository extends ServiceEntityRepository implements PasswordUpgraderInterface
{
    public function __construct(ManagerRegistry $registry)
    {
        parent::__construct($registry, User::class);
    }

    /**
     * Used to upgrade (rehash) the user's password automatically over time.
     */
    public function upgradePassword(PasswordAuthenticatedUserInterface $user, string $newHashedPassword): void
    {
        if (!$user instanceof User) {
            throw new UnsupportedUserException(sprintf('Instances of "%s" are not supported.', $user::class));
        }

        $user->setPassword($newHashedPassword);
        $this->getEntityManager()->persist($user);
        $this->getEntityManager()->flush();
    }

    /**
     * @throws NonUniqueResultException
     */
    public function findOneByEmail(array $array) : ?User
    {
        return $this->createQueryBuilder('u')
            ->andWhere('u.email = :val')
            ->setParameter('val', $array['email'])
            ->getQuery()
            ->getOneOrNullResult();
    }
    public function findAllActiveEducatorsAndAdmins() : array
    {
        $roleEducator = User::ROLE_EDUCATOR;
        $roleAdmin = User::ROLE_ADMIN;
        return $this->createQueryBuilder('u')
            ->andWhere('(u.roles LIKE :roleEducator OR u.roles LIKE :roleAdmin)')
            ->andWhere('u.active = :active')
            ->setParameter('roleEducator', "%$roleEducator%")
            ->setParameter('roleAdmin', "%$roleAdmin%")
            ->setParameter('active', true)
            ->getQuery()
            ->getResult();
    }
    public function getQueryBuilderActiveEducators() : array
    {
        $roleEducator = User::ROLE_EDUCATOR;
        $roleAdmin = User::ROLE_ADMIN;
        return $this->createQueryBuilder('u')
            ->andWhere('(u.roles LIKE :roleEducator OR u.roles LIKE :roleAdmin)')
            ->andWhere('u.active = :active')
            ->setParameter('roleEducator', "%$roleEducator%")
            ->setParameter('roleAdmin', "%$roleAdmin%")
            ->setParameter('active', true)
            ->getQuery()
            ->getResult();
    }

    public function getQueryBuilderForReceiverType(string $receiverType, User $user, ?Educator $educator): QueryBuilder
    {
        $roleEducator = User::ROLE_EDUCATOR;
        $roleAdmin = User::ROLE_ADMIN;
        $roleStudent = User::ROLE_STUDENT;
        $roleGuardian = User::ROLE_GUARDIAN;
        $q = $this->createQueryBuilder('u')
            ->andWhere('u.active = :active')
            ->andWhere('u.id != :userId')
            ->setParameter('userId', $user->getId())
            ->setParameter('active', true);
        switch ($receiverType) {
            case Message::ALL_TYPE: {
                $q->andWhere('(u.roles LIKE :roleEducator OR u.roles LIKE :roleStudent OR u.roles LIKE :roleGuardian OR u.roles LIKE :roleAdmin)')
                    ->setParameter('roleEducator', "%$roleEducator%")
                    ->setParameter('roleAdmin', "%$roleAdmin%")
                    ->setParameter('roleStudent', "%$roleStudent%")
                    ->setParameter('roleGuardian', "%$roleGuardian%");
                break;
            }
            case Message::EDUCATOR_TYPE: {
                $q->andWhere('(u.roles LIKE :roleEducator)')
                    ->setParameter('roleEducator', "%$roleEducator%");
                break;
            }
            case Message::STUDENT_TYPE: {
                $q->andWhere('(u.roles LIKE :roleStudent)')
                    ->setParameter('roleStudent', "%$roleStudent%");
                break;
            }
            case Message::GUARDIAN_TYPE: {
                $q->andWhere('(u.roles LIKE :roleGuardian)')
                    ->setParameter('roleGuardian', "%$roleGuardian%");
                break;
            }
            case Message::STUDENT_AND_GUARDIAN_TYPE: {
                $q->andWhere('(u.roles LIKE :roleStudent OR u.roles LIKE :roleGuardian)')
                    ->setParameter('roleStudent', "%$roleStudent%")
                    ->setParameter('roleGuardian', "%$roleGuardian%");
                break;
            }
            case Message::EDUCATOR_STUDENTS_TYPE: {
                $q->andWhere('(u.roles LIKE :roleStudent)')
                    ->leftJoin('u.students', 'students')
                    ->andWhere('students.educator = :educator')
                    ->setParameter('educator', $educator)
                    ->setParameter('roleStudent', "%$roleStudent%");
                break;
            }
        }
        return $q;
    }

    public function findReceiversByType(string $type, User $user, ?Educator $educator)
    {
        return $this->getQueryBuilderForReceiverType($type, $user, $educator)
            ->getQuery()
            ->getResult();
    }


}
