<?php

namespace App\Security;

use App\Entity\User;
use App\Repository\UserRepository;
use App\Services\AuthenticationRedirect;
use Symfony\Component\HttpFoundation\RedirectResponse;
use Symfony\Component\HttpFoundation\Request;
use Symfony\Component\HttpFoundation\Response;
use Symfony\Component\Routing\RouterInterface;
use Symfony\Component\Security\Core\Authentication\Token\TokenInterface;
use Symfony\Component\Security\Core\Exception\AuthenticationException;
use Symfony\Component\Security\Core\Exception\UserNotFoundException;
use Symfony\Component\Security\Http\Authenticator\AbstractAuthenticator;
use Symfony\Component\Security\Http\Authenticator\Passport\Badge\CsrfTokenBadge;
use Symfony\Component\Security\Http\Authenticator\Passport\Badge\RememberMeBadge;
use Symfony\Component\Security\Http\Authenticator\Passport\Badge\UserBadge;
use Symfony\Component\Security\Http\Authenticator\Passport\Credentials\PasswordCredentials;
use Symfony\Component\Security\Http\Authenticator\Passport\Passport;
use Symfony\Component\Security\Http\SecurityRequestAttributes;
use Symfony\Component\Security\Http\Util\TargetPathTrait;

class LoginFormAuthenticator extends AbstractAuthenticator
{
    use TargetPathTrait;
    private AuthenticationRedirect $authenticationRedirect;
    private UserRepository $userRepository;
    private RouterInterface $router;

    public function __construct(AuthenticationRedirect $authenticationRedirect, RouterInterface $router, UserRepository $userRepository)
    {
        $this->authenticationRedirect = $authenticationRedirect;
        $this->userRepository = $userRepository;
        $this->router = $router;
    }
    public function supports(Request $request): ?bool
    {
        return ($request->getPathInfo() === '/' && $request->isMethod('POST'));
    }

    public function authenticate(Request $request): Passport
    {

        $email = $request->request->get('_username');
        $password = $request->request->get('_password');

        return new Passport(
            new UserBadge($email, function($userIdentifier) {
                // optionally pass a callback to load the User manually
                $user = $this->userRepository->findOneByEmail(['email' => $userIdentifier]);
                if (!$user || $user->isActive() === false) {
                    throw new UserNotFoundException();
                }
                if (!(in_array(User::ROLE_GUARDIAN, $user->getRoles()) ||in_array(User::ROLE_STUDENT, $user->getRoles()) ||
                    in_array(User::ROLE_ADMIN, $user->getRoles()) ||in_array(User::ROLE_EDUCATOR, $user->getRoles()))) {
                    throw new UserNotFoundException();
                }
                return $user;
            }),
            new PasswordCredentials($password),
            [
                new CsrfTokenBadge(
                    'authenticate',
                    $request->request->get('_csrf_token')
                ),
                new RememberMeBadge(),
            ]
        );
    }

    public function onAuthenticationSuccess(Request $request, TokenInterface $token, string $firewallName): ?Response
    {
        if ($targetPath = $this->getTargetPath($request->getSession(), $firewallName)) {
            return new RedirectResponse($targetPath);
        }
        /** @var User $user */
        $user = $token->getUser();
        return $this->authenticationRedirect->getRedirectResponse($user);
    }

    public function onAuthenticationFailure(Request $request, AuthenticationException $exception): ?Response
    {
        $request->getSession()->set(SecurityRequestAttributes::AUTHENTICATION_ERROR, $exception);
        return new RedirectResponse(
            $this->router->generate('app_login')
        );
    }
}