<?php

namespace App\Services;


use App\Entity\Attendance;
use App\Entity\Student;
use App\Repository\AttendanceRepository;
use App\Repository\StudentRepository;
use DateTimeImmutable;
use Doctrine\ORM\NonUniqueResultException;

class EveningCheckService
{

    private StudentRepository $studentRepository;
    private AttendanceRepository $attendanceRepository;

    public function __construct(StudentRepository $studentRepository, AttendanceRepository $attendanceRepository)
    {
        $this->studentRepository = $studentRepository;
        $this->attendanceRepository = $attendanceRepository;
    }

    /**
     * @throws NonUniqueResultException
     */
    public function checkAllStudentsHaveEveningCheck(): bool
    {
        $date = new DateTimeImmutable();
        $students = $this->studentRepository->findActiveStudentsWithRoom();

        // Check if all students have evening check
        foreach ($students as $student) {
            if ($this->attendanceRepository->findEveningCheckForStudent($student, $date) === null) {
                return false;
            }
        }
        return true;
    }

    /**
     * @throws NonUniqueResultException
     */
    public function getAbsentStudentsForDate(DateTimeImmutable $date) : array
    {
        $absentStudents = [];
        $activeStudents = $this->studentRepository->findActiveStudentsWithRoom();

        /** @var Student $student */
        foreach ($activeStudents as $student) {
            $attendance = $this->attendanceRepository->findLastPresentOrAbsent($student, $date);
            if ($attendance !== null && $attendance->getAction() === Attendance::ABSENT) {
                $absentStudents[$student->getRoom()->getDoorNumber()][] = $student;
            }
        }
        ksort($absentStudents);
        return $absentStudents;
    }

    public function countAllActiveStudents(): int
    {
        $activeStudents = $this->studentRepository->findActiveStudentsWithRoom();
        return count($activeStudents);
    }

    public function getStudentsWithoutEveningCheckForDate(DateTimeImmutable $date): array
    {
        $studentsWithoutEveningCheck = [];
        $activeStudents = $this->studentRepository->findActiveStudentsWithRoom();
        $eveningChecks = $this->attendanceRepository->findEveningCheck($activeStudents, $date);
        /** @var Student $student */
        foreach ($activeStudents as $student) {
            $has_check = false;
            /** @var Attendance $eveningCheck */
            foreach ($eveningChecks as $eveningCheck) {
                if ($eveningCheck->getStudent()->getId() === $student->getId()){
                    $has_check = true;
                    break;
                }
            }
            if (!$has_check){
                $studentsWithoutEveningCheck[$student->getRoom()->getDoorNumber()][] = $student;
            }
        }
        ksort($studentsWithoutEveningCheck);
        return $studentsWithoutEveningCheck;
    }
}