<?php

namespace App\Services\Import;

use App\Entity\Meeting;
use App\Entity\MeetingAttendance;
use App\Repository\EducatorRepository;
use DateTimeImmutable;
use Doctrine\ORM\EntityManagerInterface;
use Exception;
use Symfony\Component\Form\Exception\TransformationFailedException;
use Symfony\Component\Validator\Validator\ValidatorInterface;

class ImportGroupMeetings extends Import
{

    const DATE_KEY = 'Datum';
    const ROOM_KEY = 'Miestnost';
    const EDUCATOR_KEY = 'Vychovavatel';
    private EntityManagerInterface $entityManager;
    private EducatorRepository $educatorRepository;
    private ValidatorInterface $validator;

    public function __construct(EntityManagerInterface $entityManager, ValidatorInterface $validator,
                                EducatorRepository $educatorRepository)
    {
        $this->entityManager = $entityManager;
        $this->educatorRepository = $educatorRepository;
        $this->validator = $validator;
    }

    /**
     * @throws Exception
     */
    public function saveMeeting(array $importedGroupMeetings): void
    {
        foreach ($importedGroupMeetings as $importedGroupMeeting) {
            $educator = $this->educatorRepository->findOneByEmail($importedGroupMeeting[self::EDUCATOR_KEY]);
            if ($educator === null) {
                throw new TransformationFailedException('', 0, null, 'import.wrong_educator_email', [
                    '%email%' => $importedGroupMeeting[self::EDUCATOR_KEY],
                ]);
            }

            $meeting = new Meeting();
            $meeting->setDoorNumber((int)$importedGroupMeeting[self::ROOM_KEY]);
            $meeting->setPlannedDate(new DateTimeImmutable($importedGroupMeeting[self::DATE_KEY]));
            $meeting->setEducator($educator);
            $errors = $this->validator->validate($meeting);
            if ($errors->count() > 0) {
                $errorMessages = [];
                foreach ($errors as $error) {
                    $errorMessages[] = $error->getMessage();
                }
                throw new TransformationFailedException('', 0, null, 'import.validation_error.meeting', [
                    '%error%' => implode(', ', $errorMessages),
                    '%meeting_date%' => $meeting->getPlannedDate()->format('Y-m-d H:i'),
                ]);
            }
            $this->entityManager->persist($meeting);

            $students = $educator->getStudents();
            foreach ($students as $student) {
                $meetingAttendance = new MeetingAttendance();
                $meetingAttendance->setMeeting($meeting);
                $meetingAttendance->setStudent($student);
                $errors = $this->validator->validate($meetingAttendance);
                if ($errors->count() > 0) {
                    $errorMessages = [];
                    foreach ($errors as $error) {
                        $errorMessages[] = $error->getMessage();
                    }
                    throw new TransformationFailedException('', 0, null, 'import.validation_error.meeting', [
                        '%error%' => implode(', ', $errorMessages),
                        '%meeting_date%' => $meeting->getPlannedDate()->format('Y-m-d H:i'),
                    ]);
                }
                $this->entityManager->persist($meetingAttendance);
            }
            $this->entityManager->flush();
        }
    }
}