<?php

namespace App\Services\Import;

use App\Entity\Meeting;
use App\Entity\MeetingAttendance;
use App\Repository\EducatorRepository;
use App\Repository\UserRepository;
use DateTimeImmutable;
use Doctrine\ORM\EntityManagerInterface;
use Exception;
use Symfony\Component\Form\Exception\TransformationFailedException;
use Symfony\Component\Validator\Validator\ValidatorInterface;

class ImportIndividualMeetings extends Import
{
    const DATE_KEY = 'Datum';
    const ROOM_KEY = 'Miestnost';
    const EDUCATOR_KEY = 'Vychovavatel';
    const STUDENT_KEY = 'Ziak';
    private EntityManagerInterface $entityManager;
    private UserRepository $userRepository;
    private EducatorRepository $educatorRepository;
    private ValidatorInterface $validator;

    public function __construct(EntityManagerInterface $entityManager, ValidatorInterface $validator,
                                UserRepository $userRepository, EducatorRepository $educatorRepository)
    {
        $this->entityManager = $entityManager;
        $this->userRepository = $userRepository;
        $this->educatorRepository = $educatorRepository;
        $this->validator = $validator;
    }

    /**
     * @throws Exception
     */
    public function saveMeeting(array $importedIndividualMeetings): void
    {
        foreach ($importedIndividualMeetings as $importedIndividualMeeting) {
            $educator = $this->educatorRepository->findOneByEmail($importedIndividualMeeting[self::EDUCATOR_KEY]);
            if ($educator === null) {
                throw new TransformationFailedException('', 0, null, 'import.wrong_educator_email', [
                    '%email%' => $importedIndividualMeeting[self::EDUCATOR_KEY],
                ]);
            }

            $user = $this->userRepository->findOneBy(['email' => $importedIndividualMeeting[self::STUDENT_KEY]]);
            if ($user === null || $user->getStudent() === null) {
                throw new TransformationFailedException('', 0, null, 'import.wrong_student_email', [
                    '%email%' => $importedIndividualMeeting[self::STUDENT_KEY],
                ]);
            }

            $student = $user->getStudent();
            $meeting = new Meeting();
            $meeting->setDoorNumber((int)$importedIndividualMeeting[self::ROOM_KEY]);
            $meeting->setPlannedDate(new DateTimeImmutable($importedIndividualMeeting[self::DATE_KEY]));
            $meeting->setEducator($educator);

            $meetingAttendance = new MeetingAttendance();
            $meetingAttendance->setMeeting($meeting);
            $meetingAttendance->setStudent($student);

            $errors = $this->validator->validate($meetingAttendance);
            if ($errors->count() > 0) {
                $errorMessages = [];
                foreach ($errors as $error) {
                    $errorMessages[] = $error->getMessage();
                }
                throw new TransformationFailedException('', 0, null, 'import.validation_error.meeting', [
                    '%error%' => implode(', ', $errorMessages),
                    '%meeting_date%' => $meeting->getPlannedDate()->format('Y-m-d H:i'),
                ]);
            }
            $errors = $this->validator->validate($meeting);
            if ($errors->count() > 0) {
                $errorMessages = [];
                foreach ($errors as $error) {
                    $errorMessages[] = $error->getMessage();
                }
                throw new TransformationFailedException('', 0, null, 'import.validation_error.meeting', [
                    '%error%' => implode(', ', $errorMessages),
                    '%meeting_date%' => $meeting->getPlannedDate()->format('Y-m-d H:i'),
                ]);
            }
            $this->entityManager->persist($meetingAttendance);
            $this->entityManager->persist($meeting);

            $this->entityManager->flush();
        }
    }
}