<?php

namespace App\Services\Import;


use App\Entity\Educator;
use App\Entity\Schedule;
use App\Repository\ScheduleRepository;
use DateTimeImmutable;
use Doctrine\ORM\EntityManagerInterface;
use Exception;
use Symfony\Component\Form\Exception\TransformationFailedException;
use Symfony\Component\Validator\Validator\ValidatorInterface;

class ImportSchedule extends Import
{
    const MONDAY_KEY = 'PO';
    const TUESDAY_KEY = 'UT';
    const WEDNESDAY_KEY = 'ST';
    const THURSDAY_KEY = 'ŠT';
    const FRIDAY_KEY = 'PI';
    const SATURDAY_KEY = 'SO';
    const SUNDAY_KEY = 'NE';
    const DAY_MAPPING = [
        self::MONDAY_KEY => 1,
        self::TUESDAY_KEY => 2,
        self::WEDNESDAY_KEY => 3,
        self::THURSDAY_KEY => 4,
        self::FRIDAY_KEY => 5,
        self::SATURDAY_KEY => 6,
        self::SUNDAY_KEY => 0,
        ];
    private EntityManagerInterface $entityManager;
    private ScheduleRepository $scheduleRepository;
    private ValidatorInterface $validator;
    private int $app_week_cycle;

    public function __construct(int $app_week_cycle, EntityManagerInterface $entityManager,
                                ScheduleRepository $scheduleRepository, ValidatorInterface $validator)
    {
        $this->entityManager = $entityManager;
        $this->scheduleRepository = $scheduleRepository;
        $this->validator = $validator;
        $this->app_week_cycle = $app_week_cycle;
    }

    /**
     * @throws Exception
     */
    public function saveSchedule(array $importedSchedules, Educator $educator): void
    {
        $this->scheduleRepository->removeSchedulesByEducator($educator);
        foreach ($importedSchedules as $week => $importedSchedule) {
            foreach ($importedSchedule as $day => $time) {
                if (!empty($time)) {
                    $daySchedule = new Schedule();
                    $daySchedule->setDayInWeek(self::DAY_MAPPING[$day]);
                    $daySchedule->setEducator($educator);
                    $daySchedule->setWeek(($week+1)%$this->app_week_cycle);

                    $daySchedule->setFromTime($time === Schedule::P_KEY? null : DateTimeImmutable::createFromFormat('H:i', Schedule::TRANSFORM_SCHEDULE[$time]['from']));
                    $daySchedule->setToTime($time === Schedule::P_KEY? null :DateTimeImmutable::createFromFormat('H:i', Schedule::TRANSFORM_SCHEDULE[$time]['to']));
                    $errors = $this->validator->validate($daySchedule);
                    if ($errors->count() > 0) {
                        $errorMessages = [];
                        foreach ($errors as $error) {
                            $errorMessages[] = $error->getMessage();
                        }
                        throw new TransformationFailedException('', 0, null, 'import.validation_error.schedule', [
                            '%error%' => implode(', ', $errorMessages),
                        ]);
                    }
                    $this->entityManager->persist($daySchedule);
                }
            }
        }
        $this->entityManager->flush();
    }
}