<?php

namespace App\Services\Import;

use App\Entity\Classroom;
use App\Entity\Coach;
use App\Entity\Guardian;
use App\Entity\MailNotification;
use App\Entity\MasterTeacher;
use App\Entity\Room;
use App\Entity\School;
use App\Entity\Student;
use App\Entity\User;
use App\Repository\ClassroomRepository;
use App\Repository\EducatorRepository;
use App\Repository\RoomRepository;
use App\Repository\SchoolRepository;
use App\Repository\UserRepository;
use App\Services\Mailer;
use DateTimeImmutable;
use Doctrine\ORM\EntityManagerInterface;
use Doctrine\ORM\NonUniqueResultException;
use Exception;
use Symfony\Component\Form\Exception\TransformationFailedException;
use Symfony\Component\Validator\Validator\ValidatorInterface;
use Symfony\Contracts\Translation\TranslatorInterface;

class ImportStudents extends Import
{
    // not all constants are called, but they are all used
    const STUDENT_EMAIL_KEY = 'Email';
    const STUDENT_FIRST_NAME_KEY = 'Meno';
    const STUDENT_LAST_NAME_KEY = 'Priezvisko';
    const DATE_OF_BIRTH_KEY = 'Datum narodenia';
    const STREET_KEY = 'Ulica';
    const POSTAL_CODE_KEY = 'PSC';
    const CITY_KEY = 'Mesto';
    const COUNTRY_KEY = 'Krajina';
    const STUDENT_PHONE_KEY = 'Telefon';
    const STUDENT_PHONE_COUNTRY_KEY = 'Telefonna predvolba';
    const SCHOOL_KEY = 'Skola';
    const CLASS_KEY = 'Trieda';
    const EDUCATOR_KEY = 'Vychovavatel';
    const ROOM_KEY = 'Izba';
    const GUARDIAN1_EMAIL_KEY = 'Opatrovnik1 email';
    const GUARDIAN1_FIRST_NAME_KEY = 'Opatrovnik1 meno';
    const GUARDIAN1_LAST_NAME_KEY = 'Opatrovnik1 priezvisko';
    const GUARDIAN1_PHONE_KEY = 'Opatrovnik1 telefon';
    const GUARDIAN1_PHONE_COUNTRY_KEY = 'Opatrovnik1 telefonna predvolba';
    const GUARDIAN2_EMAIL_KEY = 'Opatrovnik2 email';
    const GUARDIAN2_FIRST_NAME_KEY = 'Opatrovnik2 meno';
    const GUARDIAN2_LAST_NAME_KEY = 'Opatrovnik2 priezvisko';
    const GUARDIAN2_PHONE_KEY = 'Opatrovnik2 telefon';
    const GUARDIAN2_PHONE_COUNTRY_KEY = 'Opatrovnik2 telefonna predvolba';
    const GUARDIAN3_EMAIL_KEY = 'Opatrovnik3 email';
    const GUARDIAN3_FIRST_NAME_KEY = 'Opatrovnik3 meno';
    const GUARDIAN3_LAST_NAME_KEY = 'Opatrovnik3 priezvisko';
    const GUARDIAN3_PHONE_KEY = 'Opatrovnik3 telefon';
    const GUARDIAN3_PHONE_COUNTRY_KEY = 'Opatrovnik3 telefonna predvolba';
    const COACH_EMAIL_KEY = 'Trener email';
    const COACH_FIRST_NAME_KEY = 'Trener meno';
    const COACH_LAST_NAME_KEY = 'Trener priezvisko';
    const COACH_PHONE_KEY = 'Trener telefon';
    const COACH_PHONE_COUNTRY_KEY = 'Trener telefonna predvolba';
    const MASTERTEACHER_EMAIL_KEY = 'Triedny email';
    const MASTERTEACHER_FIRST_NAME_KEY = 'Triedny meno';
    const MASTERTEACHER_LAST_NAME_KEY = 'Triedny priezvisko';
    const MASTERTEACHER_PHONE_KEY = 'Triedny telefon';
    const MASTERTEACHER_PHONE_COUNTRY_KEY = 'Triedny telefonna predvolba';

    private EntityManagerInterface $entityManager;
    private UserRepository $userRepository;
    private SchoolRepository $schoolRepository;
    private EducatorRepository $educatorRepository;
    private RoomRepository $roomRepository;
    private ClassroomRepository $classroomRepository;
    private ValidatorInterface $validator;
    private Mailer $mailer;
    private TranslatorInterface $translator;

    public function __construct(EntityManagerInterface $entityManager, ValidatorInterface $validator,
                                UserRepository $userRepository, SchoolRepository $schoolRepository, EducatorRepository $educatorRepository,
                                RoomRepository $roomRepository, ClassroomRepository $classroomRepository, Mailer $mailer, TranslatorInterface $translator)
    {
        $this->entityManager = $entityManager;
        $this->userRepository = $userRepository;
        $this->schoolRepository = $schoolRepository;
        $this->educatorRepository = $educatorRepository;
        $this->roomRepository = $roomRepository;
        $this->classroomRepository = $classroomRepository;
        $this->validator = $validator;
        $this->mailer = $mailer;
        $this->translator = $translator;
    }

    /**
     * @throws NonUniqueResultException
     * @throws Exception
     */
    public function saveStudents(array $importedStudents): void
    {
        foreach ($importedStudents as $importedStudent) {
            $educator = $this->educatorRepository->findOneByEmail($importedStudent[self::EDUCATOR_KEY]);
            if ($educator === null) {
                throw new TransformationFailedException('', 0, null, 'import.wrong_educator_email', [
                    '%email%' => $importedStudent[self::EDUCATOR_KEY],
                ]);
            }
            if (!empty($importedStudent[self::ROOM_KEY])) {
                $room = $this->roomRepository->findOneBy(['floor' => (int)$importedStudent[self::ROOM_KEY][0], 'doorNumber' => (int)$importedStudent[self::ROOM_KEY]]);
                if ($room !== null){
                    if ($room->getStudents()->count() == 2 && !$room->containsStudentEmail($importedStudent[self::STUDENT_EMAIL_KEY])) {
                        throw new TransformationFailedException('', 0, null, 'import.full_room', [
                            '%number%' => $importedStudent[self::ROOM_KEY],
                            '%student%' => $importedStudent[self::STUDENT_EMAIL_KEY],
                        ]);
                    }
                }
                else {
                    $room = new Room();
                    $room->setFloor((int)$importedStudent[self::ROOM_KEY][0]);
                    $room->setDoorNumber((int)$importedStudent[self::ROOM_KEY]);
                    $this->entityManager->persist($room);
                }
            }

            $student = $this->setPersonToUser($importedStudent, Student::class, 'Student', null);
            $student->setDateOfBirth(empty($importedStudent[self::DATE_OF_BIRTH_KEY])? null : new DateTimeImmutable($importedStudent[self::DATE_OF_BIRTH_KEY]));
            $student->setStreet(empty($importedStudent[self::STREET_KEY])? null : $importedStudent[self::STREET_KEY]);
            $student->setCity(empty($importedStudent[self::CITY_KEY])? null : $importedStudent[self::CITY_KEY]);
            $student->setPostalCode(empty($importedStudent[self::POSTAL_CODE_KEY])? null : $importedStudent[self::POSTAL_CODE_KEY]);
            $student->setCountry(empty($importedStudent[self::COUNTRY_KEY])? null : $importedStudent[self::COUNTRY_KEY]);

            if (!empty($importedStudent[self::SCHOOL_KEY]) && !empty($importedStudent[self::CLASS_KEY])) {
                $school = $this->schoolRepository->findOneBy(['name' => $importedStudent[self::SCHOOL_KEY]]);
                if ($school === null) {
                    $school = new School();
                    $school->setName($importedStudent[self::SCHOOL_KEY]);
                    $this->entityManager->persist($school);
                }
                $classroom = $this->classroomRepository->findOneBy(['name' => $importedStudent[self::CLASS_KEY], 'school' => $school]);
                if ($classroom === null) {
                    $classroom = new Classroom();
                    $classroom->setName($importedStudent[self::CLASS_KEY]);
                    $school->addClassroom($classroom);
                    $this->entityManager->persist($classroom);
                }
                $student->setClassroom($classroom);
                $errors = $this->validator->validate($classroom);
                if ($errors->count() > 0) {
                    $errorMessages = [];
                    foreach ($errors as $error) {
                        $errorMessages[] = $error->getMessage();
                    }
                    throw new TransformationFailedException('', 0, null, 'import.validation_error.classroom', [
                        '%error%' => implode(', ', $errorMessages),
                        '%classroom%' => $classroom->getName(),
                    ]);
                }
                $errors = $this->validator->validate($school);
                if ($errors->count() > 0) {
                    $errorMessages = [];
                    foreach ($errors as $error) {
                        $errorMessages[] = $error->getMessage();
                    }
                    throw new TransformationFailedException('', 0, null, 'import.validation_error.school', [
                        '%error%' => implode(', ', $errorMessages),
                        '%school%' => $school->getName(),
                    ]);
                }
            }

            if (!empty($importedStudent[self::COACH_EMAIL_KEY])) {
                $this->setPersonToUser($importedStudent, Coach::class, 'Coach', $student);
            }
            if (!empty($importedStudent[self::COACH_EMAIL_KEY])) {
                $this->setPersonToUser($importedStudent, MasterTeacher::class, 'MasterTeacher', $student);
            }

            $this->setPersonToUser($importedStudent, Guardian::class, 'Guardian1', $student);
            $this->setPersonToUser($importedStudent, Guardian::class, 'Guardian2', $student);
            $this->setPersonToUser($importedStudent, Guardian::class, 'Guardian3', $student);

            $educator->addStudent($student);
            if (isset($room)) {
                $room->addStudent($student);
                $errors = $this->validator->validate($room);
                if ($errors->count() > 0) {
                    $errorMessages = [];
                    foreach ($errors as $error) {
                        $errorMessages[] = $error->getMessage();
                    }
                    throw new TransformationFailedException('', 0, null, 'import.validation_error.room', [
                        '%error%' => implode(', ', $errorMessages),
                        '%room%' => $room->getDoorNumber(),
                    ]);
                }
            }

            $this->entityManager->flush();
        }
    }

    private function setPersonToUser($importedStudent, $class, $type, ?Student $student): ?Student
    {
        $constantType = strtoupper($type);
        if (empty($importedStudent[constant('self::'.$constantType.'_EMAIL_KEY')])) {
            return null;
        }
        $user = $this->userRepository->findOneBy(['email' => $importedStudent[constant('self::'.$constantType.'_EMAIL_KEY')]]);
        if ($user === null) {
            $user = new User();
            $this->entityManager->persist($user);
            $this->mailer->saveMailNotification($user, $this->translator->trans('email.new_user_subject'), new DateTimeImmutable(), MailNotification::NEW_USER_TYPE);
        }
        $user->setEmail($importedStudent[constant('self::'.$constantType.'_EMAIL_KEY')]);
        $user->setFirstName($importedStudent[constant('self::'.$constantType.'_FIRST_NAME_KEY')]);
        $user->setLastName($importedStudent[constant('self::'.$constantType.'_LAST_NAME_KEY')]);
        $user->setPhoneNumber(empty($importedStudent[constant('self::'.$constantType.'_PHONE_KEY')]) ? null : $importedStudent[constant('self::'.$constantType.'_PHONE_KEY')]);
        $user->setPhoneCountry(empty($importedStudent[constant('self::'.$constantType.'_PHONE_COUNTRY_KEY')]) ? null : $importedStudent[constant('self::'.$constantType.'_PHONE_COUNTRY_KEY')]);

        if ($class === Coach::class || $class === MasterTeacher::class) {
            $user->setActive(false);
        }
        $role = User::ROLE_USER;
        if ($class === Student::class) {
            $role = User::ROLE_STUDENT;
        }
        elseif ($class === Guardian::class) {
            $role = User::ROLE_GUARDIAN;
        }
        $user->setRoles([$role]);

        $getType = substr($class, strrpos($class, '\\') + 1);
        $getter = 'get' . $getType;
        $person = $user->$getter();
        if ($person === null){
            $person = new $class();
            $person->setUser($user);
            $this->entityManager->persist($person);
        }



        $setter = 'add' . $getType;
        if ($class === MasterTeacher::class) {
            $setter = 'set' . $getType;
        }
        $student?->$setter($person);

        $errors = $this->validator->validate($person);
        if ($errors->count() > 0) {
            $errorMessages = [];
            foreach ($errors as $error) {
                $errorMessages[] = $error->getMessage();
            }
            throw new TransformationFailedException('', 0, null, 'import.validation_error.user', [
                '%error%' => implode(', ', $errorMessages),
                '%user%' => $user->getEmail(),
            ]);
        }
        if ($person instanceof Student) {
            return $person;
        }
        return null;
    }
}