<?php

namespace App\Services\Import;


use App\Entity\Educator;
use App\Entity\Schedule;
use App\Entity\Student;
use App\Entity\Activity;
use App\Repository\ScheduleRepository;
use App\Repository\ActivityRepository;
use App\Repository\UserRepository;
use DateTimeImmutable;
use Doctrine\ORM\EntityManagerInterface;
use Exception;
use Symfony\Component\Form\Exception\TransformationFailedException;
use Symfony\Component\Validator\Validator\ValidatorInterface;

class ImportTraining extends Import
{
    const STUDENT_EMAIL_KEY = 'Studenti';
    const COACH_EMAIL_KEY = 'Trener email';
    const FROM_KEY = 'Od';
    const TO_KEY = 'Do';
    const DAY_KEY = 'Den';
    const EMAIL_KEY = 'Email';
    const MONDAY_KEY = 'PO';
    const TUESDAY_KEY = 'UT';
    const WEDNESDAY_KEY = 'ST';
    const THURSDAY_KEY = 'ŠT';
    const FRIDAY_KEY = 'PI';
    const SATURDAY_KEY = 'SO';
    const SUNDAY_KEY = 'NE';
    const DAY_MAPPING = [
        self::MONDAY_KEY => 1,
        self::TUESDAY_KEY => 2,
        self::WEDNESDAY_KEY => 3,
        self::THURSDAY_KEY => 4,
        self::FRIDAY_KEY => 5,
        self::SATURDAY_KEY => 6,
        self::SUNDAY_KEY => 0,
    ];
    private EntityManagerInterface $entityManager;
    private ValidatorInterface $validator;
    private UserRepository $userRepository;
    private ActivityRepository $trainingRepository;

    public function __construct(EntityManagerInterface $entityManager, UserRepository $userRepository, ActivityRepository $trainingRepository, ValidatorInterface $validator)
    {
        $this->entityManager = $entityManager;
        $this->validator = $validator;
        $this->userRepository = $userRepository;
        $this->trainingRepository = $trainingRepository;
    }

    /**
     * @throws Exception
     */
    public function saveTraining(array $importedTrainings): void
    {
        foreach ($importedTrainings as $importedTraining) {
            $training = new Activity();
            $training->setType(Activity::TRAINING);
            $user = $this->userRepository->findOneBy(['email' => $importedTraining[self::COACH_EMAIL_KEY]]);
            if ($user === null || $user->getCoach() === null) {
                throw new TransformationFailedException('', 0, null, 'import.wrong_coach_email', [
                    '%email%' => $importedTraining[self::COACH_EMAIL_KEY],
                ]);
            }
            $coach = $user->getCoach();
            $training->setCoach($coach);
            $studentsEmail = explode(',', $importedTraining[self::STUDENT_EMAIL_KEY]);

            foreach ($studentsEmail as $email) {
                $user = $this->userRepository->findOneBy(['email' => $email]);
                if ($user === null || $user->getStudent() === null) {
                    throw new TransformationFailedException('', 0, null, 'import.wrong_student_email', [
                        '%email%' => $email,
                    ]);
                }
                $student = $user->getStudent();
                $this->removeTrainingsByStudent($student);
                $training->addStudent($student);
                $student->addCoach($coach);
            }
            $startTime = DateTimeImmutable::createFromFormat('H:i', $importedTraining[self::FROM_KEY]);
            $endTime = DateTimeImmutable::createFromFormat('H:i', $importedTraining[self::TO_KEY]);
            $training->setToTime($endTime);
            $training->setFromTime($startTime);
            $training->setDayInWeek(self::DAY_MAPPING[$importedTraining[self::DAY_KEY]]);

            $errors = $this->validator->validate($training);
            if ($errors->count() > 0) {
                $errorMessages = [];
                foreach ($errors as $error) {
                    $errorMessages[] = $error->getMessage();
                }
                throw new TransformationFailedException('', 0, null, 'import.validation_error.training', [
                    '%error%' => implode(', ', $errorMessages),
                ]);
            }
            $this->entityManager->persist($training);
        }
        $this->entityManager->flush();
    }

    private function removeTrainingsByStudent(Student $student): void
    {
        $trainings = $this->trainingRepository->findTrainingsByStudents([$student]);
        foreach ($trainings as $training) {
            $this->entityManager->remove($training);
        }
    }
}